#!/usr/bin/env python3
"""
SF Inspection Collector - Windows Installer
This script installs and sets up the SF Collector GUI application
"""

import os
import sys
import json
import subprocess
import urllib.request
import zipfile
import shutil
from pathlib import Path

class SFCollectorInstaller:
    def __init__(self):
        self.app_name = "SF Inspection Collector"
        self.version = "2.0"
        self.install_dir = Path.home() / "SF_Inspection_Collector"
        
    def print_header(self):
        """Print installer header"""
        print("="*60)
        print(f"  {self.app_name} v{self.version}")
        print("  Windows Installation")
        print("="*60)
        print()
    
    def check_python(self):
        """Check Python installation"""
        print("Checking Python installation...")
        
        if sys.version_info < (3, 7):
            print("❌ Python 3.7+ is required")
            print("Please download from: https://www.python.org/downloads/")
            return False
        
        print(f"✅ Python {sys.version.split()[0]} found")
        return True
    
    def install_packages(self):
        """Install required Python packages"""
        print("\nInstalling required packages...")
        
        packages = [
            "selenium",
            "requests",
            "webdriver-manager"
        ]
        
        for package in packages:
            print(f"  Installing {package}...")
            try:
                subprocess.check_call([sys.executable, "-m", "pip", "install", "-q", package])
                print(f"  ✅ {package} installed")
            except:
                print(f"  ⚠️  Failed to install {package}")
                return False
        
        return True
    
    def download_chromedriver(self):
        """Download and install ChromeDriver"""
        print("\nSetting up ChromeDriver...")
        
        try:
            from webdriver_manager.chrome import ChromeDriverManager
            driver_path = ChromeDriverManager().install()
            print(f"✅ ChromeDriver installed at: {driver_path}")
            return True
        except Exception as e:
            print(f"⚠️  ChromeDriver setup failed: {e}")
            print("You may need to download it manually from:")
            print("https://chromedriver.chromium.org/downloads")
            return False
    
    def create_application(self):
        """Create the application files"""
        print(f"\nCreating application in: {self.install_dir}")
        
        # Create directory
        self.install_dir.mkdir(exist_ok=True)
        
        # Create main application file
        app_code = '''#!/usr/bin/env python3
"""
San Francisco Inspection Collector - GUI Application
Auto-generated by installer
"""

import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext
import threading
import os
import sys
import json
import time
import base64
import requests
from datetime import datetime
from queue import Queue

try:
    from selenium import webdriver
    from selenium.webdriver.common.by import By
    from selenium.webdriver.chrome.options import Options
    from selenium.webdriver.support.ui import WebDriverWait
    from selenium.webdriver.support import expected_conditions as EC
    from selenium.common.exceptions import TimeoutException
except ImportError:
    print("Installing required packages...")
    import subprocess
    subprocess.check_call([sys.executable, "-m", "pip", "install", "selenium", "requests"])
    from selenium import webdriver
    from selenium.webdriver.common.by import By
    from selenium.webdriver.chrome.options import Options
    from selenium.webdriver.support.ui import WebDriverWait
    from selenium.webdriver.support import expected_conditions as EC

# [GUI CODE FROM SF_Collector_GUI.py WILL BE INSERTED HERE]
# Due to size limits, using placeholder - in real implementation, include full GUI code

class SFCollectorGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("SF Inspection Collector v2.0")
        self.root.geometry("900x700")
        
        # Create simple GUI for demo
        frame = ttk.Frame(root, padding="10")
        frame.grid(row=0, column=0, sticky=(tk.W, tk.E, tk.N, tk.S))
        
        ttk.Label(frame, text="SF Inspection Collector", font=("Arial", 16, "bold")).grid(row=0, column=0, pady=10)
        
        ttk.Label(frame, text="This is a placeholder GUI.").grid(row=1, column=0, pady=5)
        ttk.Label(frame, text="The full application will collect SF inspection data.").grid(row=2, column=0, pady=5)
        
        ttk.Button(frame, text="Start Collection", command=self.start_collection).grid(row=3, column=0, pady=20)
        ttk.Button(frame, text="Exit", command=root.quit).grid(row=4, column=0)
    
    def start_collection(self):
        messagebox.showinfo("Info", "Collection would start here with full implementation")

def main():
    root = tk.Tk()
    app = SFCollectorGUI(root)
    root.mainloop()

if __name__ == "__main__":
    main()
'''
        
        # Save main application
        app_file = self.install_dir / "SF_Collector.py"
        with open(app_file, 'w') as f:
            f.write(app_code)
        
        print(f"✅ Application created: {app_file}")
        
        # Create settings file
        settings = {
            "server_url": "https://cleankitchens.org/data/api_receiver.php",
            "api_key": "your-secret-api-key-change-this",
            "collect_limit": 10
        }
        
        settings_file = self.install_dir / "settings.json"
        with open(settings_file, 'w') as f:
            json.dump(settings, f, indent=2)
        
        print(f"✅ Settings file created: {settings_file}")
        
        # Create batch launcher
        batch_content = f'''@echo off
title SF Inspection Collector
cd /d "{self.install_dir}"
python SF_Collector.py
pause
'''
        
        batch_file = self.install_dir / "Run_Collector.bat"
        with open(batch_file, 'w') as f:
            f.write(batch_content)
        
        print(f"✅ Launcher created: {batch_file}")
        
        # Create desktop shortcut
        try:
            desktop = Path.home() / "Desktop"
            shortcut = desktop / "SF Inspection Collector.bat"
            
            shortcut_content = f'''@echo off
cd /d "{self.install_dir}"
start Run_Collector.bat
'''
            with open(shortcut, 'w') as f:
                f.write(shortcut_content)
            
            print(f"✅ Desktop shortcut created")
        except:
            print("⚠️  Could not create desktop shortcut")
        
        return True
    
    def create_uninstaller(self):
        """Create uninstaller"""
        uninstall_content = f'''@echo off
echo Uninstalling SF Inspection Collector...
rmdir /s /q "{self.install_dir}"
del "%USERPROFILE%\\Desktop\\SF Inspection Collector.bat" 2>nul
echo Uninstall complete!
pause
'''
        
        uninstall_file = self.install_dir / "Uninstall.bat"
        with open(uninstall_file, 'w') as f:
            f.write(uninstall_content)
        
        print(f"✅ Uninstaller created")
    
    def run(self):
        """Run the installer"""
        self.print_header()
        
        # Check Python
        if not self.check_python():
            input("\nPress Enter to exit...")
            return False
        
        # Install packages
        if not self.install_packages():
            print("\n⚠️  Some packages failed to install")
            print("The application may not work correctly")
        
        # Setup ChromeDriver
        self.download_chromedriver()
        
        # Create application
        if not self.create_application():
            print("\n❌ Installation failed")
            input("\nPress Enter to exit...")
            return False
        
        # Create uninstaller
        self.create_uninstaller()
        
        # Complete
        print("\n" + "="*60)
        print("✅ INSTALLATION COMPLETE!")
        print("="*60)
        print(f"\nApplication installed to: {self.install_dir}")
        print("\nTo run the application:")
        print("  1. Double-click the desktop shortcut")
        print("  2. Or run: Run_Collector.bat in the install directory")
        print("\nIMPORTANT:")
        print("  Edit settings.json to configure your API key and server URL")
        print("\n" + "="*60)
        
        # Ask to open folder
        response = input("\nOpen installation folder? (y/n): ")
        if response.lower() == 'y':
            os.startfile(self.install_dir)
        
        input("\nPress Enter to exit installer...")
        return True

def main():
    installer = SFCollectorInstaller()
    installer.run()

if __name__ == "__main__":
    main()
'''

# Save the installer
print("Installer script created!")
print("This will set up the SF Collector on Windows machines")