<?php
/**
 * API Endpoint to receive San Francisco inspection data from local collector
 * Place this file in your web-accessible directory
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Configuration
$DATA_DIR = __DIR__ . '/sf_received_data/';
$LOG_FILE = __DIR__ . '/sf_receiver.log';
$API_KEY = 'your-secret-api-key-change-this'; // Change this!

// Create data directory if it doesn't exist
if (!file_exists($DATA_DIR)) {
    mkdir($DATA_DIR, 0755, true);
    mkdir($DATA_DIR . 'json/', 0755, true);
    mkdir($DATA_DIR . 'pdfs/', 0755, true);
}

// Log function
function logMessage($message) {
    global $LOG_FILE;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($LOG_FILE, "[$timestamp] $message\n", FILE_APPEND);
}

// Validate API key
function validateApiKey() {
    global $API_KEY;
    $headers = getallheaders();
    $providedKey = isset($headers['X-API-Key']) ? $headers['X-API-Key'] : '';
    
    if ($providedKey !== $API_KEY) {
        logMessage("Invalid API key attempt");
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit();
    }
}

// Main handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate API key
    validateApiKey();
    
    // Get JSON data
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        exit();
    }
    
    $action = isset($data['action']) ? $data['action'] : 'unknown';
    
    switch ($action) {
        case 'upload_inspection':
            handleInspectionUpload($data);
            break;
            
        case 'upload_pdf':
            handlePdfUpload($data);
            break;
            
        case 'batch_upload':
            handleBatchUpload($data);
            break;
            
        case 'status':
            handleStatus();
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Unknown action']);
    }
    
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Simple status check (no API key required)
    echo json_encode([
        'status' => 'ready',
        'endpoint' => 'SF Inspection Data Receiver',
        'version' => '1.0'
    ]);
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

/**
 * Handle individual inspection upload
 */
function handleInspectionUpload($data) {
    global $DATA_DIR;
    
    if (!isset($data['inspection_id']) || !isset($data['inspection_data'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing inspection data']);
        return;
    }
    
    $inspectionId = $data['inspection_id'];
    $inspectionData = $data['inspection_data'];
    
    // Save JSON data
    $jsonPath = $DATA_DIR . 'json/' . $inspectionId . '.json';
    file_put_contents($jsonPath, json_encode($inspectionData, JSON_PRETTY_PRINT));
    
    logMessage("Received inspection: $inspectionId");
    
    echo json_encode([
        'success' => true,
        'inspection_id' => $inspectionId,
        'saved_to' => $jsonPath
    ]);
}

/**
 * Handle PDF upload (base64 encoded)
 */
function handlePdfUpload($data) {
    global $DATA_DIR;
    
    if (!isset($data['inspection_id']) || !isset($data['pdf_data'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing PDF data']);
        return;
    }
    
    $inspectionId = $data['inspection_id'];
    $pdfData = base64_decode($data['pdf_data']);
    
    if ($pdfData === false) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid base64 PDF data']);
        return;
    }
    
    // Save PDF
    $pdfPath = $DATA_DIR . 'pdfs/' . $inspectionId . '.pdf';
    file_put_contents($pdfPath, $pdfData);
    
    logMessage("Received PDF for: $inspectionId (" . strlen($pdfData) . " bytes)");
    
    echo json_encode([
        'success' => true,
        'inspection_id' => $inspectionId,
        'pdf_size' => strlen($pdfData),
        'saved_to' => $pdfPath
    ]);
}

/**
 * Handle batch upload of multiple inspections
 */
function handleBatchUpload($data) {
    global $DATA_DIR;
    
    if (!isset($data['inspections']) || !is_array($data['inspections'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing inspections array']);
        return;
    }
    
    $results = [];
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($data['inspections'] as $inspection) {
        if (!isset($inspection['inspection_id'])) {
            $errorCount++;
            continue;
        }
        
        $inspectionId = $inspection['inspection_id'];
        
        // Save JSON
        $jsonPath = $DATA_DIR . 'json/' . $inspectionId . '.json';
        file_put_contents($jsonPath, json_encode($inspection, JSON_PRETTY_PRINT));
        
        // Save PDF if provided
        if (isset($inspection['pdf_data'])) {
            $pdfData = base64_decode($inspection['pdf_data']);
            if ($pdfData !== false) {
                $pdfPath = $DATA_DIR . 'pdfs/' . $inspectionId . '.pdf';
                file_put_contents($pdfPath, $pdfData);
            }
        }
        
        $results[] = $inspectionId;
        $successCount++;
    }
    
    // Save summary
    $summaryPath = $DATA_DIR . 'batch_' . date('Y-m-d_H-i-s') . '.json';
    file_put_contents($summaryPath, json_encode([
        'timestamp' => date('c'),
        'total_received' => count($data['inspections']),
        'success_count' => $successCount,
        'error_count' => $errorCount,
        'inspection_ids' => $results
    ], JSON_PRETTY_PRINT));
    
    logMessage("Batch upload: $successCount successful, $errorCount errors");
    
    echo json_encode([
        'success' => true,
        'processed' => $successCount,
        'errors' => $errorCount,
        'inspection_ids' => $results
    ]);
}

/**
 * Get status of received data
 */
function handleStatus() {
    global $DATA_DIR;
    
    $jsonFiles = glob($DATA_DIR . 'json/*.json');
    $pdfFiles = glob($DATA_DIR . 'pdfs/*.pdf');
    
    echo json_encode([
        'success' => true,
        'json_files' => count($jsonFiles),
        'pdf_files' => count($pdfFiles),
        'last_updated' => date('c', filemtime($DATA_DIR))
    ]);
}
?>