<?php

/**

 * States Directory Controller - SoberList.org

 * 

 * This file handles the logic for both the states listing and individual state views.

 * It determines which view to show based on the presence of a state_code parameter.

 */



// Get database connection

$db = Database::getConnection();



// Check if we're viewing an individual state

$state_code = isset($_GET['state_code']) ? strtoupper($_GET['state_code']) : null;



// Helper function to update query string parameters for pagination links

function update_query_string($params = []) {

    // Start with a clean slate - don't merge with $_GET

    $updated_params = [];

    

    // If we're setting page parameter, convert it to directory-page

    if (isset($params['page'])) {

        $updated_params['directory-page'] = (int)$params['page']; // Ensure it's an integer

    }

    

    // Only include other parameters if specifically needed

    // (Currently, we only need page/directory-page for pagination)

    

    // Return the query string

    return '?' . http_build_query($updated_params);

}



// Helper function to build URLs for internal links

function buildUrl($page, $params = []) {

    $url = SITE_URL . '/' . $page;

    if (!empty($params)) {

        $url .= '?' . http_build_query($params);

    }

    return $url;

}



// Get the total number of facilities (for the main states page)

$stmt = $db->prepare("SELECT COUNT(*) FROM facilities WHERE is_active = 1");

$stmt->execute();

$total_facilities = $stmt->fetchColumn();



if (!$state_code) {

    // STATES LISTING LOGIC

    

    // Get all states with facility counts, grouped by region

    $regions = [

        'Northeast' => ['CT', 'DE', 'ME', 'MD', 'MA', 'NH', 'NJ', 'NY', 'PA', 'RI', 'VT'],

        'Southeast' => ['AL', 'AR', 'FL', 'GA', 'KY', 'LA', 'MS', 'NC', 'SC', 'TN', 'VA', 'WV'],

        'Midwest' => ['IL', 'IN', 'IA', 'KS', 'MI', 'MN', 'MO', 'NE', 'ND', 'OH', 'SD', 'WI'],

        'Southwest' => ['AZ', 'NM', 'OK', 'TX'],

        'West' => ['AK', 'CA', 'CO', 'HI', 'ID', 'MT', 'NV', 'OR', 'UT', 'WA', 'WY']

    ];

    

    // Query to get states with facility counts

    $sql = "

        SELECT 

            DISTINCT s.state_code, 

            s.state_name,

            COUNT(f.id) AS facility_count

        FROM 

            facilities f

        JOIN 

            states s ON f.state_id = s.id

        WHERE 

            f.is_active = 1

        GROUP BY 

            s.state_code, s.state_name

        ORDER BY 

            s.state_name ASC

    ";

    

    $stmt = $db->prepare($sql);

    $stmt->execute();

    $states = $stmt->fetchAll(PDO::FETCH_ASSOC);

    

    // Organize states by region

    $states_by_region = [];

    foreach ($regions as $region => $region_states) {

        $states_by_region[$region] = [];

        foreach ($states as $state) {

            if (in_array($state['state_code'], $region_states)) {

                $states_by_region[$region][] = $state;

            }

        }

    }

    

    // Get popular states (top 6 by facility count)

    $popular_states = $states;

    usort($popular_states, function($a, $b) {

        return $b['facility_count'] - $a['facility_count'];

    });

    $popular_states = array_slice($popular_states, 0, 6);

    

    // Set up breadcrumbs

    $breadcrumbs = [

        ['title' => 'Home', 'url' => SITE_URL],

        ['title' => 'States Directory', 'url' => '']

    ];

    

} else {

    // INDIVIDUAL STATE FACILITIES LOGIC

    

    // Validate state code (must be 2 letter code)

    if (strlen($state_code) !== 2 || !ctype_alpha($state_code)) {

        // Invalid state code - redirect to main states page

        header("Location: /states");

        exit();

    }

    

    // Get state information

    $stmt = $db->prepare("SELECT id, state_code, state_name FROM states WHERE state_code = ?");

    $stmt->execute([$state_code]);

    $state_info = $stmt->fetch(PDO::FETCH_ASSOC);

    

    if (!$state_info) {

        // State not found - redirect to main states page

        header("Location: /states");

        exit();

    }

    

    // Get total facilities for this state

    $stmt = $db->prepare("SELECT COUNT(*) FROM facilities WHERE state_id = ? AND is_active = 1");

    $stmt->execute([$state_info['id']]);

    $total_state_facilities = $stmt->fetchColumn();

    

    // Get filter values

    $selected_city = isset($_GET['city']) ? $_GET['city'] : '';

    $selected_care_type = isset($_GET['care_type']) ? $_GET['care_type'] : '';

    

    // Build the base SQL query for facilities

    $sql = "

        SELECT DISTINCT f.*

        FROM facilities f

    ";

    

    // Add join for care type filtering if needed

    if (!empty($selected_care_type)) {

        $sql .= "

            JOIN facility_care_types fct ON f.id = fct.facility_id

        ";

    }

    

    // Add WHERE clause

    $sql .= "

        WHERE f.state_id = ? AND f.is_active = 1

    ";

    

    // Add additional filters if provided

    $params = [$state_info['id']];

    

    if (!empty($selected_city)) {

        $sql .= " AND f.city = ?";

        $params[] = $selected_city;

    }

    

    if (!empty($selected_care_type)) {

        $sql .= " AND fct.care_type_id = ?";

        $params[] = $selected_care_type;

    }

    

    // Count total filtered facilities

    $count_sql = str_replace("SELECT DISTINCT f.*", "SELECT COUNT(DISTINCT f.id)", $sql);

    $stmt = $db->prepare($count_sql);

    $stmt->execute($params);

    $total_filtered_facilities = $stmt->fetchColumn();

    

    // Pagination settings

    $facilities_per_page = 10;

    $total_pages = ceil($total_filtered_facilities / $facilities_per_page);

    

    // Use directory-page parameter for pagination

    // Also check for regular 'page' as fallback for compatibility

    $current_page = isset($_GET['directory-page']) ? max(1, intval($_GET['directory-page'])) : 

                   (isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1);

    $offset = ($current_page - 1) * $facilities_per_page;

    

    // Add sorting and pagination

    $sql .= "

        ORDER BY f.name ASC

        LIMIT {$facilities_per_page} OFFSET {$offset}

    ";

    

    // Execute the query

    $stmt = $db->prepare($sql);

    $stmt->execute($params);

    $facilities = $stmt->fetchAll(PDO::FETCH_ASSOC);

    

    // Get all cities in this state for the filter dropdown

    $stmt = $db->prepare("

        SELECT DISTINCT city 

        FROM facilities 

        WHERE state_id = ? AND is_active = 1 AND city IS NOT NULL AND city != ''

        ORDER BY city ASC

    ");

    $stmt->execute([$state_info['id']]);

    $cities = $stmt->fetchAll(PDO::FETCH_COLUMN);

    

    // Get care types for filter dropdown

    $stmt = $db->prepare("

        SELECT DISTINCT ct.id, ct.name

        FROM care_types ct

        JOIN facility_care_types fct ON ct.id = fct.care_type_id

        JOIN facilities f ON fct.facility_id = f.id

        WHERE f.state_id = ? AND f.is_active = 1

        ORDER BY ct.name ASC

    ");

    $stmt->execute([$state_info['id']]);

    $care_types = $stmt->fetchAll(PDO::FETCH_ASSOC);

    

    // Get popular cities in this state (top 8)

    $stmt = $db->prepare("

        SELECT city, COUNT(*) as count

        FROM facilities

        WHERE state_id = ? AND is_active = 1 AND city IS NOT NULL AND city != ''

        GROUP BY city

        ORDER BY count DESC, city ASC

        LIMIT 8

    ");

    $stmt->execute([$state_info['id']]);

    $popular_cities = $stmt->fetchAll(PDO::FETCH_ASSOC);

    

    // Set up breadcrumbs

    $breadcrumbs = [

        ['title' => 'Home', 'url' => SITE_URL],

        ['title' => 'States Directory', 'url' => buildUrl('states')],

        ['title' => $state_info['state_name'], 'url' => '']

    ];

}



// Helper function to get facility care types (reusing from facility.php)

function get_facility_care_types($facility_id) {

    $db = Database::getConnection();

    $stmt = $db->prepare("

        SELECT ct.id, ct.name

        FROM care_types ct

        JOIN facility_care_types fct ON ct.id = fct.care_type_id

        WHERE fct.facility_id = ?

    ");

    $stmt->execute([$facility_id]);

    return $stmt->fetchAll(PDO::FETCH_ASSOC);

}



// Helper function to get facility image (reusing from facility.php)

function getFacilityImage($facility, $care_types) {

    // Default image and alt text

    $image_path = '/assets/images/facilities/default-facility.jpg';

    $alt_text = htmlspecialchars($facility['name']) . ' - Recovery Resource';

    $width = 800;

    $height = 600;

    

    // Check if facility has a photo

    if (!empty($facility['primary_photo'])) {

        $image_path = '/assets/images/facilities/' . $facility['primary_photo'];

    } else {

        // Determine image based on care types

        $care_type_ids = array_column($care_types, 'id');

        

        if (in_array(1, $care_type_ids)) { // Detox

            $image_path = '/assets/images/facilities/detox-center.jpg';

        } elseif (in_array(2, $care_type_ids)) { // Inpatient

            $image_path = '/assets/images/facilities/inpatient-rehab.jpg';

        } elseif (in_array(3, $care_type_ids)) { // Outpatient

            $image_path = '/assets/images/facilities/outpatient-center.jpg';

        } elseif (in_array(4, $care_type_ids)) { // Sober Living

            $image_path = '/assets/images/facilities/sober-living.jpg';

        }

    }

    

    // Full URL for schema

    $full_url = SITE_URL . $image_path;

    

    return [

        'path' => $image_path,

        'full_url' => $full_url,

        'alt' => $alt_text,

        'width' => $width,

        'height' => $height

    ];

}



// Helper function to get facility slug - UPDATED to use database slug

function get_facility_slug($facility) {

    // Use the slug from the database if it exists and is not empty

    if (isset($facility['slug']) && !empty($facility['slug'])) {

        return $facility['slug'];

    }

    

    // If no slug exists in the database, generate one based on facility name and location

    $name_part = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $facility['name']));

    $name_part = trim($name_part, '-');

    

    // Add city if available

    $location_part = '';

    if (!empty($facility['city'])) {

        $city_part = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $facility['city']));

        $location_part .= '-' . $city_part;

    }

    

    // Add state code if available

    if (!empty($facility['state_id'])) {

        // Get state code from database

        $db = Database::getConnection();

        $stmt = $db->prepare("SELECT state_code FROM states WHERE id = ?");

        $stmt->execute([$facility['state_id']]);

        $state_code = $stmt->fetchColumn();

        

        if ($state_code) {

            $location_part .= '-' . strtolower($state_code);

        }

    }

    

    // Return the combined slug

    return $name_part . $location_part;

}