<?php
require_once 'auth.php';
requireLogin();

$currentPage = 'articles';

require_once '../includes/config.php';
require_once 'admin-functions.php';

// Handle delete action
if (isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['id'])) {
    if (verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        if (deleteArticle($_POST['id'])) {
            $success = "Article deleted successfully.";
        } else {
            $error = "Failed to delete article.";
        }
    } else {
        $error = "Invalid CSRF token.";
    }
}

// Get search query
$search = $_GET['search'] ?? '';
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Get articles (with search if provided)
$client = getWeaviateClient();

$whereClause = '';
if ($search) {
    $whereClause = sprintf(', where: {
        path: ["title"],
        operator: Like,
        valueText: "*%s*"
    }', addslashes($search));
}

$query = sprintf('{
    Get {
        Articles(limit: %d, offset: %d%s, sort: [{path: ["published_date"], order: desc}]) {
            _additional {
                id
            }
            title
            slug
            published_date
            establishment_name
            city
            state
            results
            auto_tags
        }
    }
}', $limit, $offset, $whereClause);

$articles = [];
try {
    $result = $client->query($query);
    if (isset($result['data']['Get']['Articles'])) {
        $articles = $result['data']['Get']['Articles'];
    }
} catch (Exception $e) {
    $error = "Failed to load articles: " . $e->getMessage();
}

$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Articles - CleanKitchens Admin</title>
    <link rel="stylesheet" href="admin-styles.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include 'includes/sidebar.php'; ?>
        
        <div class="admin-content">
            <?php include 'includes/header.php'; ?>
            
            <div class="dashboard">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                    <h1>Articles</h1>
                    <a href="/admin/new-article.php" class="btn btn-primary">+ New Article</a>
                </div>
                
                <?php if (isset($success)): ?>
                <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <!-- Search Bar -->
                <div class="card">
                    <div class="card-body">
                        <form method="GET" style="display: flex; gap: 10px;">
                            <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" 
                                   placeholder="Search articles..." class="form-control" style="flex: 1;">
                            <button type="submit" class="btn btn-primary">Search</button>
                            <?php if ($search): ?>
                            <a href="/admin/articles.php" class="btn">Clear</a>
                            <?php endif; ?>
                        </form>
                    </div>
                </div>
                
                <!-- Articles Table -->
                <div class="card">
                    <div class="card-body">
                        <?php if (empty($articles)): ?>
                        <p style="text-align: center; padding: 40px 0; color: #999;">No articles found.</p>
                        <?php else: ?>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th style="width: 40%;">Title</th>
                                    <th>Restaurant</th>
                                    <th>Location</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th style="width: 120px;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($articles as $article): ?>
                                <tr>
                                    <td>
                                        <a href="/<?= htmlspecialchars($article['slug']) ?>" target="_blank" 
                                           style="color: #2c3e50; text-decoration: none; font-weight: 500;">
                                            <?= htmlspecialchars($article['title']) ?>
                                        </a>
                                        <?php if (!empty($article['auto_tags'])): ?>
                                        <div style="margin-top: 5px;">
                                            <?php foreach (array_slice($article['auto_tags'], 0, 3) as $tag): ?>
                                            <span style="display: inline-block; background: #e9ecef; color: #495057; 
                                                       padding: 2px 8px; border-radius: 3px; font-size: 0.75rem; 
                                                       margin-right: 5px;"><?= htmlspecialchars($tag) ?></span>
                                            <?php endforeach; ?>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= htmlspecialchars($article['establishment_name'] ?? 'N/A') ?></td>
                                    <td><?= htmlspecialchars($article['city'] ?? '') ?>, <?= htmlspecialchars($article['state'] ?? '') ?></td>
                                    <td><?= date('M j, Y', strtotime($article['published_date'])) ?></td>
                                    <td>
                                        <?php 
                                        $resultLower = strtolower($article['results'] ?? '');
                                        if (strpos($resultLower, 'fail') !== false || strpos($resultLower, 'out of business') !== false): 
                                        ?>
                                            <span class="status-badge critical">Failed</span>
                                        <?php elseif (strpos($resultLower, 'pass') !== false): ?>
                                            <span class="status-badge healthy">Passed</span>
                                        <?php else: ?>
                                            <span class="status-badge warning">Warning</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div style="display: flex; gap: 8px;">
                                            <a href="/admin/edit-article.php?id=<?= urlencode($article['_additional']['id']) ?>" 
                                               class="btn-icon" title="Edit">✏️</a>
                                            <a href="/<?= htmlspecialchars($article['slug']) ?>" target="_blank" 
                                               class="btn-icon" title="View">👁️</a>
                                            <form method="POST" style="display: inline;" 
                                                  onsubmit="return confirm('Are you sure you want to delete this article?');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?= htmlspecialchars($article['_additional']['id']) ?>">
                                                <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                                                <button type="submit" class="btn-icon" style="border: none; background: none; 
                                                        cursor: pointer;" title="Delete">🗑️</button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <!-- Pagination -->
                        <div style="display: flex; justify-content: center; gap: 10px; margin-top: 30px;">
                            <?php if ($page > 1): ?>
                            <a href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>" class="btn">← Previous</a>
                            <?php endif; ?>
                            
                            <span style="padding: 10px;">Page <?= $page ?></span>
                            
                            <?php if (count($articles) === $limit): ?>
                            <a href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>" class="btn">Next →</a>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <style>
    .alert {
        padding: 12px 20px;
        border-radius: 5px;
        margin-bottom: 20px;
    }
    .alert-success {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    .alert-danger {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
    </style>
</body>
</html>