<?php
require_once 'auth.php';
requireLogin();

$currentPage = 'articles';

require_once '../includes/config.php';
require_once 'admin-functions.php';

$articleId = $_GET['id'] ?? '';
$article = null;
$success = false;

if (!$articleId) {
    header('Location: /admin/articles.php');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = "Invalid CSRF token.";
    } else {
        // Prepare article data
        $updateData = [
            'title' => $_POST['title'],
            'slug' => $_POST['slug'],
            'content' => $_POST['content'],
            'excerpt' => $_POST['excerpt'],
            'meta_description' => $_POST['meta_description'],
            'meta_keywords' => $_POST['meta_keywords'],
            'published_date' => $_POST['published_date'],
            'establishment_name' => $_POST['establishment_name'],
            'city' => $_POST['city'],
            'state' => $_POST['state'],
            'zip_code' => $_POST['zip_code'],
            'auto_tags' => array_map('trim', explode(',', $_POST['tags'])),
            'last_updated' => date('c')
        ];
        
        if (updateArticle($articleId, $updateData)) {
            $success = true;
            $successMessage = "Article updated successfully!";
        } else {
            $error = "Failed to update article.";
        }
    }
}

// Get article data
$article = getArticleById($articleId);

if (!$article) {
    header('Location: /admin/articles.php');
    exit;
}

$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Article - CleanKitchens Admin</title>
    <link rel="stylesheet" href="admin-styles.css">
    <style>
        .editor-container {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
        }
        
        .main-editor {
            background: white;
            border-radius: 8px;
            padding: 30px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .sidebar-editor {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            height: fit-content;
            position: sticky;
            top: 90px;
        }
        
        .form-section {
            margin-bottom: 30px;
        }
        
        .form-section h3 {
            margin-bottom: 15px;
            color: #2c3e50;
            font-size: 1.1rem;
            border-bottom: 1px solid #ecf0f1;
            padding-bottom: 10px;
        }
        
        .success-banner {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .error-banner {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        textarea.content-editor {
            min-height: 500px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }
        
        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .meta-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        @media (max-width: 1024px) {
            .editor-container {
                grid-template-columns: 1fr;
            }
            
            .sidebar-editor {
                position: static;
            }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <?php include 'includes/sidebar.php'; ?>
        
        <div class="admin-content">
            <?php include 'includes/header.php'; ?>
            
            <div class="dashboard">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                    <h1>Edit Article</h1>
                    <div>
                        <a href="/<?= htmlspecialchars($article['slug']) ?>" target="_blank" class="btn">View Article</a>
                        <a href="/admin/articles.php" class="btn">← Back to Articles</a>
                    </div>
                </div>
                
                <?php if ($success): ?>
                <div class="success-banner">
                    <span><?= htmlspecialchars($successMessage) ?></span>
                    <a href="/<?= htmlspecialchars($article['slug']) ?>" target="_blank" class="btn btn-sm">View Updated Article →</a>
                </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                <div class="error-banner"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                    
                    <div class="editor-container">
                        <div class="main-editor">
                            <!-- Title -->
                            <div class="form-section">
                                <div class="form-group">
                                    <label class="form-label" for="title">Article Title</label>
                                    <input type="text" id="title" name="title" class="form-control" 
                                           value="<?= htmlspecialchars($article['title'] ?? '') ?>" required>
                                </div>
                            </div>
                            
                            <!-- Slug -->
                            <div class="form-section">
                                <div class="form-group">
                                    <label class="form-label" for="slug">URL Slug</label>
                                    <input type="text" id="slug" name="slug" class="form-control" 
                                           value="<?= htmlspecialchars($article['slug'] ?? '') ?>" required>
                                    <small style="color: #7f8c8d;">URL: cleankitchens.org/<span id="slug-preview"><?= htmlspecialchars($article['slug'] ?? '') ?></span></small>
                                </div>
                            </div>
                            
                            <!-- Content -->
                            <div class="form-section">
                                <h3>Article Content</h3>
                                <div class="form-group">
                                    <textarea id="content" name="content" class="form-control content-editor" required><?= htmlspecialchars($article['content'] ?? '') ?></textarea>
                                </div>
                            </div>
                            
                            <!-- SEO Section -->
                            <div class="form-section">
                                <h3>SEO & Meta Data</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="excerpt">Excerpt (Brief Summary)</label>
                                    <textarea id="excerpt" name="excerpt" class="form-control" rows="3"><?= htmlspecialchars($article['excerpt'] ?? '') ?></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="meta_description">Meta Description (SEO)</label>
                                    <textarea id="meta_description" name="meta_description" class="form-control" rows="2" 
                                              maxlength="160"><?= htmlspecialchars($article['meta_description'] ?? '') ?></textarea>
                                    <small style="color: #7f8c8d;">Max 160 characters for search engines</small>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="meta_keywords">Meta Keywords</label>
                                    <input type="text" id="meta_keywords" name="meta_keywords" class="form-control" 
                                           value="<?= htmlspecialchars($article['meta_keywords'] ?? '') ?>">
                                    <small style="color: #7f8c8d;">Comma-separated keywords</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="sidebar-editor">
                            <!-- Publishing -->
                            <div class="form-section">
                                <h3>Publishing</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="published_date">Published Date</label>
                                    <input type="datetime-local" id="published_date" name="published_date" class="form-control" 
                                           value="<?= date('Y-m-d\TH:i', strtotime($article['published_date'] ?? 'now')) ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label">Last Updated</label>
                                    <p style="margin: 5px 0; color: #7f8c8d;">
                                        <?= date('M j, Y g:i A', strtotime($article['last_updated'] ?? 'now')) ?>
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Tags -->
                            <div class="form-section">
                                <h3>Tags</h3>
                                <div class="form-group">
                                    <textarea id="tags" name="tags" class="form-control" rows="3"><?= htmlspecialchars(implode(', ', $article['auto_tags'] ?? [])) ?></textarea>
                                    <small style="color: #7f8c8d;">Comma-separated tags</small>
                                </div>
                            </div>
                            
                            <!-- Restaurant Info -->
                            <div class="form-section">
                                <h3>Restaurant Information</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="establishment_name">Restaurant Name</label>
                                    <input type="text" id="establishment_name" name="establishment_name" class="form-control" 
                                           value="<?= htmlspecialchars($article['establishment_name'] ?? '') ?>">
                                </div>
                                
                                <div class="meta-grid">
                                    <div class="form-group">
                                        <label class="form-label" for="city">City</label>
                                        <input type="text" id="city" name="city" class="form-control" 
                                               value="<?= htmlspecialchars($article['city'] ?? '') ?>">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label" for="state">State</label>
                                        <input type="text" id="state" name="state" class="form-control" 
                                               value="<?= htmlspecialchars($article['state'] ?? '') ?>" maxlength="2">
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="zip_code">ZIP Code</label>
                                    <input type="text" id="zip_code" name="zip_code" class="form-control" 
                                           value="<?= htmlspecialchars($article['zip_code'] ?? '') ?>">
                                </div>
                            </div>
                            
                            <!-- Actions -->
                            <div class="button-group">
                                <button type="submit" class="btn btn-primary" style="flex: 1;">Update Article</button>
                                <a href="/admin/articles.php" class="btn" style="flex: 1;">Cancel</a>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        // Update slug preview
        document.getElementById('slug').addEventListener('input', function() {
            document.getElementById('slug-preview').textContent = this.value;
        });
        
        // Auto-save draft (optional)
        let autoSaveTimer;
        document.querySelectorAll('input, textarea').forEach(element => {
            element.addEventListener('input', function() {
                clearTimeout(autoSaveTimer);
                autoSaveTimer = setTimeout(() => {
                    console.log('Would auto-save here...');
                }, 3000);
            });
        });
    </script>
</body>
</html>