<?php
require_once 'auth.php';
requireLogin();

$currentPage = 'new-article';

require_once '../includes/config.php';
require_once 'admin-functions.php';

$success = false;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = "Invalid CSRF token.";
    } else {
        // Prepare article data
        $articleData = [
            'title' => $_POST['title'],
            'slug' => $_POST['slug'] ?: generateSlug($_POST['title']),
            'content' => $_POST['content'],
            'excerpt' => $_POST['excerpt'],
            'meta_description' => $_POST['meta_description'],
            'meta_keywords' => $_POST['meta_keywords'],
            'published_date' => $_POST['published_date'] ?: date('c'),
            'establishment_name' => $_POST['establishment_name'],
            'city' => $_POST['city'],
            'state' => $_POST['state'],
            'zip_code' => $_POST['zip_code'],
            'auto_tags' => array_map('trim', explode(',', $_POST['tags'])),
            'image_url' => $_POST['image_url'] ?: '/assets/images/violations/general_1.jpg',
            'ai_generated' => false,
            'manually_created' => true
        ];
        
        $articleId = createArticle($articleData);
        if ($articleId) {
            $success = true;
            $successMessage = "Article created successfully!";
            header("Location: /admin/edit-article.php?id=$articleId&created=1");
            exit;
        } else {
            $error = "Failed to create article.";
        }
    }
}

$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>New Article - CleanKitchens Admin</title>
    <link rel="stylesheet" href="admin-styles.css">
    <style>
        .editor-container {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
        }
        
        .main-editor {
            background: white;
            border-radius: 8px;
            padding: 30px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .sidebar-editor {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            height: fit-content;
            position: sticky;
            top: 90px;
        }
        
        .form-section {
            margin-bottom: 30px;
        }
        
        .form-section h3 {
            margin-bottom: 15px;
            color: #2c3e50;
            font-size: 1.1rem;
            border-bottom: 1px solid #ecf0f1;
            padding-bottom: 10px;
        }
        
        textarea.content-editor {
            min-height: 500px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }
        
        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .meta-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        .image-selector {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin-top: 10px;
        }
        
        .image-option {
            cursor: pointer;
            border: 2px solid transparent;
            border-radius: 5px;
            padding: 5px;
            text-align: center;
            font-size: 0.75rem;
        }
        
        .image-option:hover {
            border-color: #667eea;
        }
        
        .image-option.selected {
            border-color: #667eea;
            background: #f0f4ff;
        }
        
        @media (max-width: 1024px) {
            .editor-container {
                grid-template-columns: 1fr;
            }
            
            .sidebar-editor {
                position: static;
            }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <?php include 'includes/sidebar.php'; ?>
        
        <div class="admin-content">
            <?php include 'includes/header.php'; ?>
            
            <div class="dashboard">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                    <h1>Create New Article</h1>
                    <a href="/admin/articles.php" class="btn">← Back to Articles</a>
                </div>
                
                <?php if (isset($error)): ?>
                <div class="error-banner"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                    
                    <div class="editor-container">
                        <div class="main-editor">
                            <!-- Title -->
                            <div class="form-section">
                                <div class="form-group">
                                    <label class="form-label" for="title">Article Title *</label>
                                    <input type="text" id="title" name="title" class="form-control" required>
                                </div>
                            </div>
                            
                            <!-- Slug -->
                            <div class="form-section">
                                <div class="form-group">
                                    <label class="form-label" for="slug">URL Slug (leave blank to auto-generate)</label>
                                    <input type="text" id="slug" name="slug" class="form-control">
                                    <small style="color: #7f8c8d;">URL: cleankitchens.org/<span id="slug-preview"></span></small>
                                </div>
                            </div>
                            
                            <!-- Content -->
                            <div class="form-section">
                                <h3>Article Content *</h3>
                                <div class="form-group">
                                    <textarea id="content" name="content" class="form-control content-editor" required 
                                              placeholder="Write your article content here..."></textarea>
                                </div>
                            </div>
                            
                            <!-- SEO Section -->
                            <div class="form-section">
                                <h3>SEO & Meta Data</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="excerpt">Excerpt (Brief Summary)</label>
                                    <textarea id="excerpt" name="excerpt" class="form-control" rows="3"></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="meta_description">Meta Description (SEO)</label>
                                    <textarea id="meta_description" name="meta_description" class="form-control" rows="2" 
                                              maxlength="160"></textarea>
                                    <small style="color: #7f8c8d;">Max 160 characters for search engines</small>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="meta_keywords">Meta Keywords</label>
                                    <input type="text" id="meta_keywords" name="meta_keywords" class="form-control">
                                    <small style="color: #7f8c8d;">Comma-separated keywords</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="sidebar-editor">
                            <!-- Publishing -->
                            <div class="form-section">
                                <h3>Publishing</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="published_date">Published Date</label>
                                    <input type="datetime-local" id="published_date" name="published_date" class="form-control" 
                                           value="<?= date('Y-m-d\TH:i') ?>">
                                </div>
                            </div>
                            
                            <!-- Featured Image -->
                            <div class="form-section">
                                <h3>Featured Image</h3>
                                <div class="form-group">
                                    <input type="hidden" id="image_url" name="image_url" value="/assets/images/violations/general_1.jpg">
                                    <div class="image-selector">
                                        <div class="image-option selected" data-url="/assets/images/violations/general_1.jpg">
                                            📋 General
                                        </div>
                                        <div class="image-option" data-url="/assets/images/violations/temperature_1.jpg">
                                            🌡️ Temperature
                                        </div>
                                        <div class="image-option" data-url="/assets/images/violations/handwashing_1.jpg">
                                            🧼 Handwashing
                                        </div>
                                        <div class="image-option" data-url="/assets/images/violations/pest_1.jpg">
                                            🐛 Pest
                                        </div>
                                        <div class="image-option" data-url="/assets/images/violations/sanitation_1.jpg">
                                            🧹 Sanitation
                                        </div>
                                        <div class="image-option" data-url="/assets/images/violations/closure_1.jpg">
                                            🚫 Closure
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Tags -->
                            <div class="form-section">
                                <h3>Tags</h3>
                                <div class="form-group">
                                    <textarea id="tags" name="tags" class="form-control" rows="3" 
                                              placeholder="Restaurant, Chicago, Health Inspection"></textarea>
                                    <small style="color: #7f8c8d;">Comma-separated tags</small>
                                </div>
                            </div>
                            
                            <!-- Restaurant Info -->
                            <div class="form-section">
                                <h3>Restaurant Information</h3>
                                
                                <div class="form-group">
                                    <label class="form-label" for="establishment_name">Restaurant Name</label>
                                    <input type="text" id="establishment_name" name="establishment_name" class="form-control">
                                </div>
                                
                                <div class="meta-grid">
                                    <div class="form-group">
                                        <label class="form-label" for="city">City</label>
                                        <input type="text" id="city" name="city" class="form-control" value="Chicago">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label" for="state">State</label>
                                        <input type="text" id="state" name="state" class="form-control" value="IL" maxlength="2">
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="zip_code">ZIP Code</label>
                                    <input type="text" id="zip_code" name="zip_code" class="form-control">
                                </div>
                            </div>
                            
                            <!-- Actions -->
                            <div class="button-group">
                                <button type="submit" class="btn btn-primary" style="flex: 1;">Create Article</button>
                                <a href="/admin/articles.php" class="btn" style="flex: 1;">Cancel</a>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        // Update slug preview
        document.getElementById('slug').addEventListener('input', function() {
            document.getElementById('slug-preview').textContent = this.value || 'auto-generated-slug';
        });
        
        // Auto-generate slug from title
        document.getElementById('title').addEventListener('blur', function() {
            if (!document.getElementById('slug').value) {
                const slug = this.value.toLowerCase()
                    .replace(/[^a-z0-9]+/g, '-')
                    .replace(/^-|-$/g, '');
                const datePrefix = new Date().toISOString().slice(5,10).replace('-','') + 
                                 new Date().getFullYear().toString().slice(-2);
                document.getElementById('slug-preview').textContent = datePrefix + '-' + slug;
            }
        });
        
        // Image selector
        document.querySelectorAll('.image-option').forEach(option => {
            option.addEventListener('click', function() {
                document.querySelectorAll('.image-option').forEach(o => o.classList.remove('selected'));
                this.classList.add('selected');
                document.getElementById('image_url').value = this.dataset.url;
            });
        });
    </script>
</body>
</html>