#!/usr/bin/env python3
"""
Test with Firefox and human-like behavior
Adds mouse movements, random delays, and more realistic browsing patterns
"""

import time
import subprocess
import os
import random
from selenium import webdriver
from selenium.webdriver.firefox.options import Options
from selenium.webdriver.firefox.service import Service
from selenium.webdriver.common.by import By
from selenium.webdriver.common.action_chains import ActionChains
from webdriver_manager.firefox import GeckoDriverManager
import logging

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

class HumanBrowser:
    def __init__(self):
        self.xvfb_proc = None
        self.display = ":99"
        self.driver = None
        
    def start_xvfb(self):
        """Start Xvfb virtual display"""
        try:
            # Kill any existing Xvfb
            subprocess.run(["pkill", "-f", "Xvfb :99"], capture_output=True)
            time.sleep(1)
            
            logger.info("Starting Xvfb...")
            self.xvfb_proc = subprocess.Popen([
                "Xvfb", self.display,
                "-screen", "0", "1920x1080x24",
                "-ac", "+extension", "GLX", "+render", "-noreset"
            ], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
            
            os.environ["DISPLAY"] = self.display
            time.sleep(2)
            
            logger.info(f"Xvfb started on {self.display}")
            return True
            
        except Exception as e:
            logger.error(f"Failed to start Xvfb: {e}")
            return False
    
    def human_delay(self, min_sec=0.5, max_sec=2.0):
        """Random human-like delay"""
        delay = random.uniform(min_sec, max_sec)
        time.sleep(delay)
    
    def setup_firefox(self):
        """Setup Firefox with human-like settings"""
        try:
            firefox_options = Options()
            
            # Don't use headless - we're using Xvfb
            # firefox_options.add_argument("--headless")
            
            # Set preferences to appear more human
            firefox_options.set_preference("dom.webdriver.enabled", False)
            firefox_options.set_preference("useAutomationExtension", False)
            firefox_options.set_preference("general.useragent.override", 
                "Mozilla/5.0 (X11; Linux x86_64; rv:109.0) Gecko/20100101 Firefox/115.0")
            
            # Disable various Firefox features that might reveal automation
            firefox_options.set_preference("dom.webnotifications.enabled", False)
            firefox_options.set_preference("geo.enabled", False)
            firefox_options.set_preference("media.navigator.enabled", False)
            firefox_options.set_preference("network.http.use-cache", True)
            
            # Set window size
            firefox_options.set_preference("browser.window.width", 1920)
            firefox_options.set_preference("browser.window.height", 1080)
            
            logger.info("Starting Firefox browser...")
            
            # Use webdriver-manager to get geckodriver
            service = Service(GeckoDriverManager().install())
            self.driver = webdriver.Firefox(service=service, options=firefox_options)
            
            # Set window size
            self.driver.set_window_size(1920, 1080)
            
            # Set realistic timeouts
            self.driver.set_page_load_timeout(30)
            self.driver.implicitly_wait(10)
            
            logger.info("Firefox browser started successfully")
            return True
            
        except Exception as e:
            logger.error(f"Failed to setup Firefox: {e}")
            return False
    
    def human_mouse_movement(self):
        """Simulate human-like mouse movements"""
        try:
            actions = ActionChains(self.driver)
            
            # Move mouse in a somewhat random pattern
            for _ in range(random.randint(2, 4)):
                x = random.randint(100, 1800)
                y = random.randint(100, 900)
                
                # Move to position with random speed
                actions.move_by_offset(x, y)
                self.human_delay(0.1, 0.3)
            
            actions.perform()
            
        except Exception as e:
            logger.debug(f"Mouse movement failed: {e}")
    
    def scroll_page(self):
        """Simulate human scrolling"""
        try:
            # Scroll down a bit
            scroll_amount = random.randint(100, 300)
            self.driver.execute_script(f"window.scrollBy(0, {scroll_amount});")
            self.human_delay(0.5, 1.5)
            
        except Exception as e:
            logger.debug(f"Scroll failed: {e}")
    
    def test_access(self):
        """Try to access the site with human-like behavior"""
        try:
            url = "https://inspections.myhealthdepartment.com/san-francisco"
            
            # First visit a neutral site (Google)
            logger.info("Visiting Google first (like a human would)...")
            self.driver.get("https://www.google.com")
            self.human_delay(2, 4)
            
            # Do some mouse movements
            self.human_mouse_movement()
            
            # Now navigate to target site
            logger.info(f"Now navigating to target: {url}")
            self.driver.get(url)
            
            # Wait like a human would
            self.human_delay(3, 5)
            
            # Simulate reading the page
            self.human_mouse_movement()
            self.scroll_page()
            
            # Get page info
            current_url = self.driver.current_url
            page_title = self.driver.title
            
            logger.info(f"Current URL: {current_url}")
            logger.info(f"Page Title: {page_title}")
            
            # Take screenshot
            screenshot_path = "/tmp/sf_firefox_screenshot.png"
            self.driver.save_screenshot(screenshot_path)
            logger.info(f"Screenshot saved to: {screenshot_path}")
            
            # Save page source
            page_source = self.driver.page_source
            with open("/tmp/sf_firefox_page.html", "w") as f:
                f.write(page_source)
            logger.info("Page source saved to: /tmp/sf_firefox_page.html")
            
            # Check if blocked
            if "403" in page_title or "Forbidden" in page_source[:500]:
                logger.warning("⚠️  BLOCKED: Got 403 Forbidden with Firefox too")
                
                # Try one more thing - execute JavaScript
                logger.info("Trying JavaScript navigation...")
                self.driver.execute_script(f"window.location.href = '{url}';")
                self.human_delay(3, 5)
                
                # Check again
                page_title = self.driver.title
                if "403" not in page_title:
                    logger.info("✅ JavaScript navigation worked!")
                    return True
                
                return False
            else:
                logger.info("✅ SUCCESS: Page loaded without 403!")
                
                # Look for content
                try:
                    links = self.driver.find_elements(By.TAG_NAME, "a")
                    logger.info(f"Found {len(links)} links on page")
                    
                    # Check for inspection links
                    inspection_links = []
                    for link in links:
                        href = link.get_attribute("href") or ""
                        if "inspectionID=" in href:
                            inspection_links.append(href)
                    
                    if inspection_links:
                        logger.info(f"✅ Found {len(inspection_links)} inspection links!")
                        for i, link in enumerate(inspection_links[:3]):
                            logger.info(f"  Inspection {i+1}: {link}")
                    
                except Exception as e:
                    logger.warning(f"Error examining content: {e}")
                
                return True
                
        except Exception as e:
            logger.error(f"Error accessing site: {e}")
            return False
    
    def cleanup(self):
        """Clean up browser and Xvfb"""
        try:
            if self.driver:
                logger.info("Closing Firefox...")
                self.driver.quit()
            
            if self.xvfb_proc:
                logger.info("Stopping Xvfb...")
                self.xvfb_proc.terminate()
                self.xvfb_proc.wait(timeout=5)
                
        except Exception as e:
            logger.error(f"Error during cleanup: {e}")
            subprocess.run(["pkill", "-f", "Xvfb"], capture_output=True)
            subprocess.run(["pkill", "-f", "firefox"], capture_output=True)

def main():
    logger.info("="*60)
    logger.info("SF SITE ACCESS - FIREFOX WITH HUMAN BEHAVIOR")
    logger.info("="*60)
    
    browser = HumanBrowser()
    
    try:
        # Start virtual display
        if not browser.start_xvfb():
            logger.error("Failed to start Xvfb")
            return
        
        # Setup Firefox
        if not browser.setup_firefox():
            logger.error("Failed to setup Firefox")
            return
        
        # Test access
        success = browser.test_access()
        
        # Report results
        print("\n" + "="*60)
        print("FIREFOX TEST RESULTS")
        print("="*60)
        
        if success:
            print("✅ SUCCESS with Firefox!")
            print("\nNext steps:")
            print("1. Parse the inspection links")
            print("2. Click each inspection")
            print("3. Extract data and download PDFs")
        else:
            print("❌ Still blocked even with Firefox")
            print("\nRemaining options:")
            print("1. Use a real browser profile with cookies")
            print("2. Try using a proxy service")
            print("3. Use Playwright instead of Selenium")
            print("4. Manual cookie extraction and injection")
        
        print("\nFiles created:")
        print("- /tmp/sf_firefox_screenshot.png")
        print("- /tmp/sf_firefox_page.html")
        
    except Exception as e:
        logger.error(f"Unexpected error: {e}")
        import traceback
        traceback.print_exc()
        
    finally:
        browser.cleanup()
        
    logger.info("Test complete")

if __name__ == "__main__":
    main()