<?php

/**

 * Facility Page Logic

 * 

 * This file contains the logic for the facility page.

 */



// Error handling setup

ini_set('display_errors', 0);

ini_set('display_startup_errors', 0);

error_reporting(E_ALL);

ini_set('log_errors', 1);

ini_set('error_log', BASE_PATH . '/logs/facility-error.log');



// Required includes

require_once(BASE_PATH . '/includes/init.php');

require_once(BASE_PATH . '/includes/functions.php');



// Prevent config.php from being included here

if (defined('CONFIG_INCLUDED')) {

    // Config already included

} else {

    define('CONFIG_INCLUDED', true);

}



// Check if slug is provided

if (empty($slug)) {

    header("HTTP/1.1 404 Not Found");

    echo "Facility not found";

    exit;

}



// Debug mode flag

$debug_mode = isset($_GET['debug']) && ($_GET['debug'] === 'reviews' || $_GET['debug'] === 'force-update');



if (!function_exists('generate_facility_url')) {

    function generate_facility_url($facility) {

        return $facility['slug'] ?? 

               strtolower(

                   preg_replace('/-+/', '-', 

                       preg_replace('/[^a-zA-Z0-9\s-]/', '', $facility['name']) . '-' . 

                       $facility['city'] . '-' . 

                       ($facility['state_code'] ?? '')

                   )

               );

    }

}



// IMPORTANT: Added function_exists check to prevent redeclaration

if (!function_exists('getFacilityImage')) {

    function getFacilityImage($facility, $care_types) {

        $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') 

            . $_SERVER['HTTP_HOST'];

        

        $default_image = '/assets/images/default-facility.jpg';

        $image_info = [

            'path' => $default_image,

            'alt' => htmlspecialchars($facility['name']) . ' - Recovery Resource',

            'full_url' => $base_url . $default_image,

            'width' => 400, // Set to the actual image dimensions

            'height' => 400  // Set to the actual image dimensions

        ];

        

        $has_HH = false;

        $has_SA = false;

        $has_DT = false;

        

        if (!empty($care_types)) {

            foreach ($care_types as $type) {

                $care_type_code = $type['code'] ?? '';

                if ($care_type_code === 'HH') $has_HH = true;

                elseif ($care_type_code === 'SA') $has_SA = true;

                elseif ($care_type_code === 'DT') $has_DT = true;

            }

        }

        

        $name_lower = strtolower($facility['name'] ?? '');

        $desc_lower = strtolower($facility['description'] ?? '');

        

        $has_house_keywords = (strpos($name_lower, 'house') !== false || 

                              strpos($name_lower, 'sober') !== false || 

                              strpos($name_lower, 'transitional') !== false || 

                              strpos($desc_lower, 'sober living') !== false || 

                              strpos($desc_lower, 'sober house') !== false || 

                              strpos($desc_lower, 'transitional') !== false || 

                              strpos($desc_lower, 'halfway house') !== false);

        

        $has_treatment_keywords = (strpos($name_lower, 'treatment') !== false || 

                                 strpos($name_lower, 'rehab') !== false || 

                                 strpos($desc_lower, 'treatment center') !== false || 

                                 strpos($desc_lower, 'rehabilitation') !== false || 

                                 strpos($desc_lower, 'rehab') !== false);

        

        $has_detox_keywords = (strpos($name_lower, 'detox') !== false || 

                             strpos($desc_lower, 'detox') !== false || 

                             strpos($desc_lower, 'detoxification') !== false);

        

        if ($has_HH || $has_house_keywords) {

            $image_path = '/assets/images/sober-house-near-me.webp';

            $alt_text = htmlspecialchars($facility['name']) . ' - Sober Living Home';

            $image_info['width'] = 400;

            $image_info['height'] = 400;

            

            $jpg_fallback = '/assets/images/sober-house-near-me.jpg';

            $png_fallback = '/assets/images/sober-house-near-me.png';

        } 

        elseif ($has_SA || $has_treatment_keywords) {

            $image_path = '/assets/images/treatment-near-me.webp';

            $alt_text = htmlspecialchars($facility['name']) . ' - Treatment Center';

            $image_info['width'] = 400;

            $image_info['height'] = 400;

            

            $jpg_fallback = '/assets/images/treatment-near-me.jpg';

            $png_fallback = '/assets/images/treatment-near-me.png';

        } 

        elseif ($has_DT || $has_detox_keywords) {

            $image_path = '/assets/images/detox-near-me.webp';

            $alt_text = htmlspecialchars($facility['name']) . ' - Detox Facility';

            $image_info['width'] = 400;

            $image_info['height'] = 400;

            

            $jpg_fallback = '/assets/images/detox-near-me.jpg';

            $png_fallback = '/assets/images/detox-near-me.png';

        } 

        else {

            $image_path = '/assets/images/we-do-recover.webp';

            $alt_text = htmlspecialchars($facility['name']) . ' - Recovery Resource';

            $image_info['width'] = 400;

            $image_info['height'] = 400;

            

            $jpg_fallback = '/assets/images/we-do-recover.jpg';

            $png_fallback = '/assets/images/we-do-recover.png';

        }

        

        if (file_exists($_SERVER['DOCUMENT_ROOT'] . $image_path)) {

            $image_info['path'] = $image_path;

            $image_info['alt'] = $alt_text;

            $image_info['full_url'] = $base_url . $image_path;

        }

        elseif (file_exists($_SERVER['DOCUMENT_ROOT'] . $jpg_fallback)) {

            $image_info['path'] = $jpg_fallback;

            $image_info['alt'] = $alt_text;

            $image_info['full_url'] = $base_url . $jpg_fallback;

        }

        elseif (file_exists($_SERVER['DOCUMENT_ROOT'] . $png_fallback)) {

            $image_info['path'] = $png_fallback;

            $image_info['alt'] = $alt_text;

            $image_info['full_url'] = $base_url . $png_fallback;

        }

        

        return $image_info;

    }

}



// Add function_exists check to all other functions

if (!function_exists('determineFacilityPrimaryType')) {

    function determineFacilityPrimaryType($facility, $care_types) {

        // First check care types if available

        if (!empty($care_types)) {

            foreach ($care_types as $type) {

                $care_type_code = $type['code'] ?? '';

                if ($care_type_code === 'HH') return 'HH';

                if ($care_type_code === 'DT') return 'DT';

                if ($care_type_code === 'SA') return 'SA';

            }

        }

        

        // Fall back to name/description keyword checking

        $name_lower = strtolower($facility['name'] ?? '');

        $desc_lower = strtolower($facility['description'] ?? '');

        

        if (strpos($name_lower, 'house') !== false || 

            strpos($name_lower, 'sober') !== false || 

            strpos($name_lower, 'transitional') !== false || 

            strpos($desc_lower, 'sober living') !== false || 

            strpos($desc_lower, 'sober house') !== false || 

            strpos($desc_lower, 'transitional') !== false || 

            strpos($desc_lower, 'halfway house') !== false) {

            return 'HH';

        }

        

        if (strpos($name_lower, 'detox') !== false || 

            strpos($desc_lower, 'detox') !== false || 

            strpos($desc_lower, 'detoxification') !== false) {

            return 'DT';

        }

        

        if (strpos($name_lower, 'treatment') !== false || 

            strpos($name_lower, 'rehab') !== false || 

            strpos($desc_lower, 'treatment center') !== false || 

            strpos($desc_lower, 'rehabilitation') !== false || 

            strpos($desc_lower, 'rehab') !== false) {

            return 'SA';

        }

        

        // Default to treatment if we can't determine

        return 'SA';

    }

}



if (!function_exists('updateFacilityReviews')) {

    function updateFacilityReviews($facility_id, $place_id) {

        if (empty($facility_id) || empty($place_id)) {

            return false;

        }

        

        $pdo = get_db_connection();

        

        try {

            $checkStmt = $pdo->prepare("

                SELECT 

                    (reviews_cache_check_date IS NULL 

                    OR reviews_cache_check_date < DATE_FORMAT(CURRENT_DATE, '%Y-%m-01')) as should_update,

                    last_reviews_update

                FROM facilities

                WHERE id = ?

            ");

            $checkStmt->execute([$facility_id]);

            $updateCheck = $checkStmt->fetch(PDO::FETCH_ASSOC);

            

            if (empty($updateCheck['should_update'])) {

                return false;

            }

            

            $reviews = fetchGooglePlacesReviews($place_id);

            

            if (empty($reviews)) {

                return false;

            }

            

            $reviews_json = json_encode($reviews);

            

            // Use force_update_facility_reviews function if available (from test-review.php)

            if (function_exists('force_update_facility_reviews')) {

                return force_update_facility_reviews($facility_id, $reviews_json);

            } else {

                $updateStmt = $pdo->prepare("

                    UPDATE facilities 

                    SET 

                        google_reviews = ?,

                        last_reviews_update = NOW(),

                        reviews_cache_check_date = CURRENT_DATE

                    WHERE id = ?

                ");

                

                return $updateStmt->execute([$reviews_json, $facility_id]);

            }

        } catch (Exception $e) {

            error_log("Exception updating reviews for facility $facility_id: " . $e->getMessage());

            return false;

        }

    }

}



if (!function_exists('check_facility_reviews_cache')) {

    function check_facility_reviews_cache($facility_id) {

        if (empty($facility_id)) {

            return ['needs_update' => false, 'error' => 'Invalid facility ID'];

        }

        

        $pdo = get_db_connection();

        

        try {

            $stmt = $pdo->prepare("

                SELECT 

                    (reviews_cache_check_date IS NULL 

                    OR reviews_cache_check_date < DATE_FORMAT(CURRENT_DATE, '%Y-%m-01')) as needs_update,

                    last_reviews_update,

                    google_reviews as cached_reviews

                FROM facilities

                WHERE id = ?

            ");

            

            $stmt->execute([$facility_id]);

            return $stmt->fetch(PDO::FETCH_ASSOC);

        } catch (Exception $e) {

            error_log("Error checking facility reviews cache: " . $e->getMessage());

            return ['needs_update' => false, 'error' => $e->getMessage()];

        }

    }

}



if (!function_exists('displayReviewsFromJson')) {

    function displayReviewsFromJson($reviews, $last_update = null) {

        $html = '';

        

        if (!empty($last_update)) {

            $html .= '<div class="text-muted small mb-3">Last updated: ' . date('F j, Y', strtotime($last_update)) . '</div>';

        }

        

        $average_rating = 0;

        $total_reviews = count($reviews);

        

        if ($total_reviews > 0) {

            $sum = 0;

            foreach ($reviews as $review) {

                $sum += $review['rating'] ?? 0;

            }

            $average_rating = $sum / $total_reviews;

        }

        

        if ($average_rating > 0) {

            $html .= '<div class="review-summary mb-4">

                <div class="d-flex align-items-center">

                    <div class="h1 mb-0 me-3">' . number_format($average_rating, 1) . '</div>

                    <div>

                        <div class="stars">';

            

            for ($i = 1; $i <= 5; $i++) {

                if ($i <= floor($average_rating)) {

                    $html .= '<i class="bi bi-star-fill" style="color: gold;"></i> ';

                } elseif ($i - 0.5 <= $average_rating) {

                    $html .= '<i class="bi bi-star-half" style="color: gold;"></i> ';

                } else {

                    $html .= '<i class="bi bi-star" style="color: gold;"></i> ';

                }

            }

            

            $html .= '</div>

                        <div class="text-muted">' . $total_reviews . ' ' . ($total_reviews === 1 ? 'review' : 'reviews') . '</div>

                    </div>

                </div>

            </div>';

        }

        

        $html .= '<div class="reviews-list">';

        

        if (empty($reviews)) {

            return $html . '<p class="text-muted py-3">No Google reviews available.</p></div>';

        }

        

        foreach ($reviews as $review) {

            if (empty($review['author_name'])) continue;

            

            $reviewDate = !empty($review['time']) ? date('F j, Y', is_numeric($review['time']) ? $review['time'] : strtotime($review['time'])) : '';

            

            $html .= '

                <div class="review card p-3 mb-3">

                    <div class="d-flex justify-content-between align-items-center mb-2">

                        <div class="d-flex align-items-center">

                            ' . (!empty($review['profile_photo_url']) ? 

                                '<img src="' . htmlspecialchars($review['profile_photo_url']) . '" alt="' . htmlspecialchars($review['author_name']) . '" class="review-profile-image me-2" width="50" height="50">' : '') . '

                            <div>

                                <strong>' . htmlspecialchars($review['author_name']) . '</strong>

                                ' . ($reviewDate ? '<div class="text-muted small">' . $reviewDate . '</div>' : '') . '

                            </div>

                        </div>

                        <div class="ratings">';

            

            for ($i = 1; $i <= 5; $i++) {

                $html .= '<i class="bi bi-star' . ($i <= ($review['rating'] ?? 0) ? '-fill' : '') . '" style="color: ' . ($i <= ($review['rating'] ?? 0) ? 'gold' : '#ccc') . ';"></i>';

            }

            

            $html .= '

                        </div>

                    </div>

                    ' . (!empty($review['text']) ? '<p class="mb-0">' . htmlspecialchars($review['text']) . '</p>' : '') . '

                </div>

            ';

        }

        

        $html .= '</div>';

        

        return $html;

    }

}



// Define force_update_facility_reviews function if it doesn't exist

if (!function_exists('force_update_facility_reviews')) {

    function force_update_facility_reviews($facility_id, $reviews_json) {

        $pdo = get_db_connection();

        

        // Ensure we have a database connection

        if (!$pdo) {

            error_log("No database connection available in function");

            return false;

        }

        

        // Update facility reviews

        $stmt = $pdo->prepare("

            UPDATE facilities 

            SET 

                google_reviews = ?,

                last_reviews_update = NOW(),

                reviews_cache_check_date = CURRENT_DATE

            WHERE id = ?

        ");

        

        $result = $stmt->execute([$reviews_json, $facility_id]);

        

        return $result;

    }

}



try {

    // Get facility by slug

    $facility = get_facility_by_slug($slug);

    

    if (!$facility) {

        header("HTTP/1.1 404 Not Found");

        echo "Facility not found";

        exit;

    }

    

    // Force-update reviews if in debug mode

    if (isset($_GET['debug']) && $_GET['debug'] === 'force-update') {

        // Create test review data

        $test_reviews = [

            [

                "author_name" => "Test User",

                "rating" => 5,

                "text" => "Test review created on " . date('Y-m-d H:i:s')

            ]

        ];

        

        // Convert to JSON

        $reviews_json = json_encode($test_reviews);

        

        // Use the force_update_facility_reviews function if available

        if (function_exists('force_update_facility_reviews')) {

            $result = force_update_facility_reviews($facility['id'], $reviews_json);

            

            if (!$result) {

                // Try direct SQL if function fails

                $pdo = get_db_connection();

                if ($pdo) {

                    $stmt = $pdo->prepare("

                        UPDATE facilities 

                        SET 

                            google_reviews = ?,

                            last_reviews_update = NOW(),

                            reviews_cache_check_date = CURRENT_DATE

                        WHERE id = ?

                    ");

                    

                    $result = $stmt->execute([$reviews_json, $facility['id']]);

                }

            }

        }

    }

    

    // Check if reviews need updating

    if (!empty($facility['place_id'])) {

        $reviewsCache = check_facility_reviews_cache($facility['id']);

        

        if (!empty($reviewsCache['needs_update'])) {

            updateFacilityReviews($facility['id'], $facility['place_id']);

        }

    }

    

    // Get facility related data

    $photos = get_facility_photos($facility['id']);

    $care_types = get_facility_care_types($facility['id']);

    $service_settings = get_facility_service_settings($facility['id']);

    $payment_options = get_facility_payment_options($facility['id']);

    $special_programs = get_facility_special_programs($facility['id']);

    $support_services = get_facility_support_services($facility['id']);

    $similar_facilities = get_similar_facilities($facility['id'], $facility['state_id'], $facility['city'], 3);



    // Determine primary facility type for schema

    $facility_primary_type = determineFacilityPrimaryType($facility, $care_types);



    // Collect all facility images

    $all_images = [];

    

    if (!empty($photos)) {

        foreach ($photos as $photo) {

            if (!empty($photo['file_name'])) {

                $all_images[] = [

                    'src' => FACILITY_PHOTOS_URL . $photo['file_name'],

                    'type' => 'local'

                ];

            }

        }

    }

    

    for ($i = 1; $i <= 5; $i++) {

        $img_field = ($i === 1) ? 'image_url' : "img_url-" . ($i-1);

        if (!empty($facility[$img_field])) {

            $all_images[] = [

                'src' => $facility[$img_field],

                'type' => 'google'

            ];

        }

    }



    // Get facility reviews

    $reviews = get_facility_reviews($facility['id'], 10);

    $google_reviews = [];

    $google_rating = 0;

    $google_reviews_count = 0;

    

    if (!empty($facility['google_reviews'])) {

        $google_reviews = json_decode($facility['google_reviews'], true) ?: [];

        

        if (!empty($google_reviews)) {

            $sum = 0;

            foreach ($google_reviews as $review) {

                $sum += $review['rating'] ?? 0;

            }

            $google_rating = count($google_reviews) > 0 ? $sum / count($google_reviews) : 0;

            $google_reviews_count = count($google_reviews);

        }

    }



    // Get facility image

    $image_info = getFacilityImage($facility, $care_types);

    

    // If there are Google reviews, add them to the facility object for schema

    if (!empty($google_reviews) && $google_rating > 0) {

        $facility['avg_rating'] = $google_rating;

        $facility['total_reviews'] = $google_reviews_count;

        

        $facility['reviews'] = [];

        $maxReviews = min(5, count($google_reviews));

        for ($i = 0; $i < $maxReviews; $i++) {

            $review = $google_reviews[$i];

            if (!empty($review['author_name']) && !empty($review['text'])) {

                $facility['reviews'][] = [

                    'author' => $review['author_name'],

                    'rating' => $review['rating'] ?? 0,

                    'comment' => substr($review['text'], 0, 250),

                    'date' => date('Y-m-d', !empty($review['time']) ? (is_numeric($review['time']) ? $review['time'] : strtotime($review['time'])) : time())

                ];

            }

        }

    }

    

    // Add care types to facility object for schema

    if (!empty($care_types)) {

        $facility['care_types'] = $care_types;

    }

    

    // Set up base URL for canonical links

    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'];



    // The canonical URL must be properly defined before it's used in breadcrumbs

    $canonical_url = $base_url . '/facility/' . generate_facility_url($facility);

    

    // Set breadcrumbs for the page

    $breadcrumbs = [

        ['title' => 'Home', 'url' => SITE_URL],

        ['title' => 'Directory', 'url' => SITE_URL . '/directory'],

    ];

    

    // Add state to breadcrumbs if available

    if (!empty($facility['state_code'])) {

        $breadcrumbs[] = [

            'title' => $facility['state_name'],

            'url' => SITE_URL . '/directory?state=' . $facility['state_code']

        ];

        

        // Add city to breadcrumbs if available

        if (!empty($facility['city'])) {

            $breadcrumbs[] = [

                'title' => $facility['city'],

                'url' => SITE_URL . '/directory?state=' . $facility['state_code'] . '&city=' . urlencode($facility['city'])

            ];

        }

    }

    

    // Add current facility as the last breadcrumb - Using canonical URL properly

    $breadcrumbs[] = [

        'title' => $facility['name'], 

        'url' => $canonical_url

    ];

    

} catch (Exception $e) {

    error_log("Error fetching facility details: " . $e->getMessage(), 3, BASE_PATH . '/logs/facility-error.log');

    header("HTTP/1.1 500 Internal Server Error");

    echo "An unexpected error occurred while retrieving facility details.";

    exit;

}



/**

 * Generate keywords based on facility data and care types

 * 

 * @param array $facility Facility data

 * @param array $care_types Care types

 * @return string Comma-separated keywords

 */

function generateKeywords($facility, $care_types) {

    $keywords = [];

    

    // Add facility name

    $keywords[] = $facility['name'];

    

    // Add location-based keywords

    if (!empty($facility['city']) && !empty($facility['state_code'])) {

        $keywords[] = "recovery " . $facility['city'] . " " . $facility['state_code'];

        $keywords[] = "addiction treatment " . $facility['city'];

        $keywords[] = "rehab " . $facility['city'];

        $keywords[] = "rehab near me " . $facility['city'];

        $keywords[] = "substance abuse treatment " . $facility['city'] . " " . $facility['state_code'];

    }

    

    // Add care type keywords

    if (!empty($care_types)) {

        foreach ($care_types as $type) {

            $keywords[] = $type['name'];

            if (!empty($facility['city'])) {

                $keywords[] = $type['name'] . " " . $facility['city'];

                $keywords[] = $type['name'] . " near me " . $facility['city'];

            }

        }

    }

    

    // Add generic recovery terms

    $generic_terms = [

        "recovery", "addiction treatment", "rehabilitation", "sober living", 

        "substance abuse treatment", "addiction help", "recovery resources",

        "drug rehab", "alcohol rehab", "drug treatment center", "alcohol treatment",

        "drug addiction treatment", "detox center", "sober living home",

        "inpatient rehab", "outpatient rehab", "addiction recovery"

    ];

    

    $keywords = array_merge($keywords, $generic_terms);

    

    // Remove duplicates and join

    return implode(", ", array_unique($keywords));

}



/**

 * Get additional type URLs based on facility type for schema.org

 * 

 * @param string $facility_type Facility type code

 * @return array Array of schema.org type URLs

 */

function getAdditionalTypeUrls($facility_type) {

    $type_urls = [

        // Base URLs for all facilities

        "https://schema.org/LocalBusiness",

        "https://schema.org/HealthAndBeautyBusiness"

    ];

    

    switch ($facility_type) {

        case 'SA':

            $type_urls[] = "https://schema.org/MedicalBusiness";

            $type_urls[] = "https://health-lifesci.schema.org/MedicalClinic";

            $type_urls[] = "https://schema.org/Hospital";

            break;

        

        case 'DT':

            $type_urls[] = "https://schema.org/MedicalClinic";

            $type_urls[] = "https://health-lifesci.schema.org/MedicalClinic";

            $type_urls[] = "https://schema.org/EmergencyService";

            break;

            

        case 'HH':

            $type_urls[] = "https://schema.org/LodgingBusiness";

            $type_urls[] = "https://schema.org/Residence";

            $type_urls[] = "https://schema.org/Apartment";

            break;

            

        case 'SUMH':

        case 'MH':

            $type_urls[] = "https://schema.org/MedicalClinic";

            $type_urls[] = "https://health-lifesci.schema.org/MedicalClinic";

            $type_urls[] = "https://schema.org/PsychologicalTreatment";

            break;

            

        case 'SHELTER':

            $type_urls[] = "https://schema.org/EmergencyService";

            $type_urls[] = "https://schema.org/NonprofitOrganization";

            $type_urls[] = "https://schema.org/HousingComplex";

            break;

    }

    

    return $type_urls;

}



/**

 * Generate meta tags and schema.org structured data for a facility

 * 

 * @param array $facility Facility data

 * @param array $care_types Care types for the facility

 * @param string $canonical_url Canonical URL

 * @param array $image_info Image information

 * @param bool $output Whether to output the schema directly (true) or return it (false)

 * @return string|void If $output is false, returns the schema as JSON, otherwise outputs directly

 */

function generateMetaAndSchema($facility, $care_types, $canonical_url, $image_info, $output = true) {

    // Facility Type Mapping to Schema - Enhanced with more specific types

    $facility_type_map = [

        'SA' => 'MedicalBusiness', // Substance use treatment centers

        'DT' => 'MedicalClinic', // Detoxification centers

        'HH' => 'LodgingBusiness', // Sober homes/Transitional housing

        'SUMH' => 'MedicalClinic', // Co-occurring disorder treatment centers

        'MH' => 'MedicalClinic', // Mental health treatment facilities

        'SHELTER' => 'EmergencyService' // Emergency housing

    ];

    

    // Determine Facility Type

    $facility_primary_type = determineFacilityPrimaryType($facility, $care_types);

    $schema_type = $facility_type_map[$facility_primary_type] ?? 'MedicalOrganization';

    

    // Build base ID for knowledge graph connections

    $base_id = $canonical_url . '#';

    

    // Prepare address for schema

    $address = [

        "@type" => "PostalAddress",

        "streetAddress" => !empty($facility['address_line1']) ? htmlspecialchars($facility['address_line1']) : "12 G St",

        "addressLocality" => !empty($facility['city']) ? htmlspecialchars($facility['city']) : "Hampton",

        "addressRegion" => !empty($facility['state_code']) ? htmlspecialchars($facility['state_code']) : "NH",

        "postalCode" => !empty($facility['zip_code']) ? htmlspecialchars($facility['zip_code']) : "03842",

        "addressCountry" => "US"

    ];

    

    // Prepare phone for schema

    $phone = !empty($facility['phone']) ? htmlspecialchars($facility['phone']) : "(888) 501-7071";

    

    // Prepare additional type URLs based on facility type

    $additional_type_urls = getAdditionalTypeUrls($facility_primary_type);

    

    // **Main Business/Organization Schema** - Enhanced with Knowledge Graph connections

    $localBusinessSchema = [

        "@context" => "https://schema.org",

        "@type" => $schema_type,

        "@id" => $base_id . "organization",

        "name" => htmlspecialchars($facility['name']),

        "url" => $canonical_url,

        "sameAs" => [

            // Add website if available

            !empty($facility['website']) ? $facility['website'] : null,

        ],

        "image" => [

            "@type" => "ImageObject",

            "@id" => $base_id . "primaryImage",

            "url" => $image_info['full_url'],

            "width" => $image_info['width'],

            "height" => $image_info['height'],

            "caption" => $image_info['alt']

        ],

        "description" => htmlspecialchars(strip_tags($facility['description'] ?? '')),

        "address" => $address,

        "telephone" => $phone,

        "email" => "info@soberlist.org",

        "additionalType" => $additional_type_urls,

        "isAccessibleForFree" => true,

        "keywords" => generateKeywords($facility, $care_types),

        "hasMap" => "https://www.google.com/maps/search/?api=1&query=" . urlencode(

            $facility['address_line1'] . ' ' . 

            $facility['city'] . ', ' . 

            $facility['state_code'] . ' ' . 

            $facility['zip_code']

        )

    ];

    

    // Filter out null or empty values

    $localBusinessSchema["sameAs"] = array_filter($localBusinessSchema["sameAs"]);

    

    // Add geo coordinates if available

    if (!empty($facility['latitude']) && !empty($facility['longitude'])) {

        $localBusinessSchema["geo"] = [

            "@type" => "GeoCoordinates",

            "latitude" => $facility['latitude'],

            "longitude" => $facility['longitude']

        ];

    }

    

    // Add opening hours if available - default to standard hours

    $localBusinessSchema["openingHoursSpecification"] = [

        [

            "@type" => "OpeningHoursSpecification",

            "dayOfWeek" => ["Monday", "Tuesday", "Wednesday", "Thursday", "Friday"],

            "opens" => "09:00",

            "closes" => "17:00"

        ]

    ];

    

    // Add specialty fields based on facility type

    switch ($facility_primary_type) {

        case 'SA': 

            // Substance use treatment centers

            $localBusinessSchema["medicalSpecialty"] = ["Addiction Medicine", "Substance Abuse Treatment"];

            $localBusinessSchema["availableService"] = [

                [

                    "@type" => "MedicalTherapy",

                    "name" => "Substance Abuse Treatment",

                    "description" => "Evidence-based approach to treating substance use disorders"

                ]

            ];

            break;

        

        case 'DT': 

            // Detox centers

            $localBusinessSchema["medicalSpecialty"] = ["Addiction Medicine", "Detoxification"];

            $localBusinessSchema["availableService"] = [

                [

                    "@type" => "MedicalTherapy",

                    "name" => "Medical Detoxification",

                    "description" => "Supervised withdrawal from substances under medical care"

                ]

            ];

            break;

            

        case 'HH':

            // Sober homes

            $localBusinessSchema["amenityFeature"] = [

                [

                    "@type" => "LocationFeatureSpecification",

                    "name" => "Substance-Free Environment",

                    "value" => true,

                    "description" => "Strictly enforced sober living environment"

                ],

                [

                    "@type" => "LocationFeatureSpecification",

                    "name" => "Peer Support",

                    "value" => true,

                    "description" => "Community-based recovery support"

                ]

            ];

            break;

    }

    

    // **AggregateRating Schema (If reviews exist)** - Fixed to handle Google reviews properly

    // Check for Google reviews from the database

    if (!empty($facility['google_reviews']) && is_string($facility['google_reviews'])) {

        $google_reviews = json_decode($facility['google_reviews'], true);

        

        if (!empty($google_reviews) && json_last_error() === JSON_ERROR_NONE) {

            // Calculate average rating and count

            $sum = 0;

            foreach ($google_reviews as $review) {

                $sum += $review['rating'] ?? 0;

            }

            

            $avg_rating = count($google_reviews) > 0 ? $sum / count($google_reviews) : 0;

            $review_count = count($google_reviews);

            

            if ($avg_rating > 0 && $review_count > 0) {

                $localBusinessSchema["aggregateRating"] = [

                    "@type" => "AggregateRating",

                    "@id" => $base_id . "aggregateRating",

                    "ratingValue" => number_format($avg_rating, 1),

                    "reviewCount" => $review_count,

                    "bestRating" => "5",

                    "worstRating" => "1"

                ];

                

                // Add individual reviews if available

                if (count($google_reviews) > 0) {

                    $localBusinessSchema["review"] = [];

                    

                    foreach ($google_reviews as $idx => $review) {

                        if ($idx >= 10) break; // Limit to 10 reviews for performance

                        

                        if (!empty($review['author_name'])) {

                            $localBusinessSchema["review"][] = [

                                "@type" => "Review",

                                "author" => [

                                    "@type" => "Person",

                                    "name" => htmlspecialchars($review['author_name'])

                                ],

                                "datePublished" => date('Y-m-d', is_numeric($review['time'] ?? time()) ? $review['time'] : time()),

                                "reviewRating" => [

                                    "@type" => "Rating",

                                    "ratingValue" => $review['rating'] ?? 5,

                                    "bestRating" => "5",

                                    "worstRating" => "1"

                                ],

                                "reviewBody" => htmlspecialchars(substr($review['text'] ?? '', 0, 250))

                            ];

                        }

                    }

                }

            }

        }

    }

    // Legacy support for facility reviews in standard format

    else if (!empty($facility['avg_rating']) && !empty($facility['total_reviews'])) {

        $localBusinessSchema["aggregateRating"] = [

            "@type" => "AggregateRating",

            "@id" => $base_id . "aggregateRating",

            "ratingValue" => number_format($facility['avg_rating'], 1),

            "reviewCount" => (int) $facility['total_reviews'],

            "bestRating" => "5",

            "worstRating" => "1"

        ];

        

        // Add individual reviews if available

        if (!empty($facility['reviews'])) {

            $localBusinessSchema["review"] = [];

            

            foreach ($facility['reviews'] as $idx => $review) {

                if ($idx >= 10) break; // Limit to 10 reviews for performance

                

                $localBusinessSchema["review"][] = [

                    "@type" => "Review",

                    "author" => [

                        "@type" => "Person",

                        "name" => htmlspecialchars($review['author'])

                    ],

                    "datePublished" => $review['date'],

                    "reviewRating" => [

                        "@type" => "Rating",

                        "ratingValue" => $review['rating'],

                        "bestRating" => "5",

                        "worstRating" => "1"

                    ],

                    "reviewBody" => htmlspecialchars($review['comment'])

                ];

            }

        }

    }

    

    // **Advisory Article Schema** - Using a more generic recovery journey article

    $articleSchema = [

        "@context" => "https://schema.org",

        "@type" => "Article",

        "@id" => $base_id . "article",

        "headline" => "Your Recovery Journey: Understanding Addiction Treatment Options",

        "author" => [

            "@type" => "Person", 

            "name" => "Chris Nesbitt",

            "@id" => SITE_URL . "#person-chris-nesbitt"

        ],

        "datePublished" => date("Y-m-d"),

        "dateModified" => date("Y-m-d"),

        "mainEntityOfPage" => [

            "@type" => "WebPage", 

            "@id" => $canonical_url

        ],

        "image" => [

            "@type" => "ImageObject",

            "url" => SITE_URL . "/assets/images/recovery-journey.jpg",

            "width" => 1200,

            "height" => 630

        ],

        "publisher" => [

            "@type" => "Organization", 

            "name" => "SoberList.org", 

            "@id" => SITE_URL . "#organization",

            "logo" => [

                "@type" => "ImageObject", 

                "url" => SITE_URL . "/logo.png",

                "width" => 180,

                "height" => 60

            ]

        ],

        "articleBody" => "Understanding the journey to recovery is essential for anyone seeking help with substance use disorders. From detoxification to aftercare, each step in the treatment process addresses different aspects of addiction recovery. This comprehensive guide helps you navigate treatment options, considerations for choosing the right facility, and what to expect during the recovery process.",

        "about" => [

            "@type" => "Thing",

            "@id" => $base_id . "organization"

        ]

    ];

    

    // **ENHANCED Breadcrumb Schema** - Improved for better Google recognition

    $breadcrumbItems = [];

    $position = 1;

    

    // Always include Home

    $breadcrumbItems[] = [

        "@type" => "ListItem",

        "position" => $position++,

        "name" => "Home",

        "item" => SITE_URL

    ];

    

    // Always include Directory

    $breadcrumbItems[] = [

        "@type" => "ListItem",

        "position" => $position++,

        "name" => "Directory",

        "item" => SITE_URL . "/directory"

    ];

    

    // State (if applicable)

    if (!empty($facility['state_code']) && !empty($facility['state_name'])) {

        $breadcrumbItems[] = [

            "@type" => "ListItem",

            "position" => $position++,

            "name" => htmlspecialchars($facility['state_name']),

            "item" => SITE_URL . "/directory?state=" . htmlspecialchars($facility['state_code'])

        ];

    }

    

    // City (if applicable)

    if (!empty($facility['city']) && !empty($facility['state_code'])) {

        $breadcrumbItems[] = [

            "@type" => "ListItem",

            "position" => $position++,

            "name" => htmlspecialchars($facility['city']),

            "item" => SITE_URL . "/directory?state=" . 

                     htmlspecialchars($facility['state_code']) . 

                     "&city=" . urlencode($facility['city'])

        ];

    }

    

    // Facility (current page always last)

    $breadcrumbItems[] = [

        "@type" => "ListItem",

        "position" => $position,

        "name" => htmlspecialchars($facility['name']),

        "item" => $canonical_url

    ];

    

    $breadcrumbSchema = [

        "@context" => "https://schema.org",

        "@type" => "BreadcrumbList",

        "@id" => $base_id . "breadcrumbs",

        "itemListElement" => $breadcrumbItems

    ];

    

    // **Organization Schema** - For establishing site identity in knowledge graph

    $organizationSchema = [

        "@context" => "https://schema.org",

        "@type" => "Organization",

        "@id" => SITE_URL . "#organization",

        "name" => "SoberList.org",

        "url" => SITE_URL,

        "logo" => [

            "@type" => "ImageObject",

            "url" => SITE_URL . "/logo.png",

            "width" => 180,

            "height" => 60

        ],

        "description" => "SoberList.org provides the most comprehensive directory of sober living homes and recovery resources across the United States.",

        "sameAs" => [

            "https://www.facebook.com/soberlist",

            "https://www.twitter.com/soberlist",

            "https://www.instagram.com/soberlist"

        ]

    ];

    

    // **WebPage Schema** - To tie everything together in the Knowledge Graph

    $webPageSchema = [

        "@context" => "https://schema.org",

        "@type" => "WebPage",

        "@id" => $canonical_url,

        "url" => $canonical_url,

        "name" => htmlspecialchars($facility['name']) . " | Recovery Resource | SoberList.org",

        "description" => htmlspecialchars(substr(strip_tags($facility['description'] ?? ''), 0, 160)) . "...",

        "isPartOf" => [

            "@type" => "WebSite",

            "@id" => SITE_URL . "#website",

            "url" => SITE_URL,

            "name" => "SoberList.org",

            "description" => "Comprehensive directory of addiction recovery resources",

            "publisher" => [

                "@id" => SITE_URL . "#organization"

            ]

        ],

        "primaryImageOfPage" => [

            "@id" => $base_id . "primaryImage"

        ],

        "breadcrumb" => [

            "@id" => $base_id . "breadcrumbs"

        ],

        "mainEntity" => [

            "@id" => $base_id . "organization"

        ],

        "about" => [

            "@id" => $base_id . "organization"

        ],

        "specialty" => "Addiction Recovery Resources",

        "datePublished" => date("Y-m-d"),

        "dateModified" => date("Y-m-d")

    ];

    

    // **Open Graph Meta Tags** - Enhanced for better social sharing - NO TRAILING SLASHES

    $og_meta_tags = '

    <meta property="og:title" content="' . htmlspecialchars($facility['name']) . '">

    <meta property="og:description" content="' . htmlspecialchars(substr(strip_tags($facility['description'] ?? ''), 0, 160)) . '...">

    <meta property="og:url" content="' . htmlspecialchars($canonical_url) . '">

    <meta property="og:image" content="' . htmlspecialchars($image_info['full_url']) . '">

    <meta property="og:image:width" content="' . $image_info['width'] . '">

    <meta property="og:image:height" content="' . $image_info['height'] . '">

    <meta property="og:image:alt" content="' . htmlspecialchars($image_info['alt']) . '">

    <meta property="og:type" content="website">

    <meta property="og:site_name" content="Sober List">

    <meta property="og:locale" content="en_US">

    <meta property="twitter:card" content="summary_large_image">

    <meta property="twitter:title" content="' . htmlspecialchars($facility['name']) . '">

    <meta property="twitter:description" content="' . htmlspecialchars(substr(strip_tags($facility['description'] ?? ''), 0, 160)) . '...">

    <meta property="twitter:image" content="' . htmlspecialchars($image_info['full_url']) . '">

    ';

    

    // Collect all schemas in an array to be returned or output

    $schemas = [

        'organization' => $organizationSchema,

        'webpage' => $webPageSchema,

        'breadcrumb' => $breadcrumbSchema,

        'localbusiness' => $localBusinessSchema,

        'article' => $articleSchema

    ];

    

    // Define schema JSON for output

    $schema_json = [];

    foreach ($schemas as $schema) {

        $schema_json[] = json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    }

    

    // If $output is true, directly output the schema JSON and meta tags

    if ($output) {

        foreach ($schema_json as $json) {

            echo '<script type="application/ld+json">' . $json . '</script>';

        }

        echo $og_meta_tags;

        return;

    }

    

    // Otherwise, return the schema data

    return [

        'schemas' => $schema_json,

        'meta_tags' => $og_meta_tags

    ];

}