<?php
// SIMPLE FUNCTIONS - No caching, no complexity

function queryWeaviate($query) {
    $url = 'http://localhost:8080/v1/graphql';
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['query' => $query]));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $response) {
        return json_decode($response, true);
    }
    
    return null;
}

function getArticleBySlug($slug) {
    $query = '{
        Get {
            Articles(where: {path: ["slug"], operator: Equal, valueText: "' . addslashes($slug) . '"}) {
                title
                slug
                content
                excerpt
                meta_description
                city
                state
                establishment_name
                published_date
                image_url
                tags
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'][0])) {
        $article = $response['data']['Get']['Articles'][0];
        
        // Set all required fields for head.php template
        $article['meta_title'] = $article['title'];
        $article['meta_description'] = $article['meta_description'] ?: $article['excerpt'];
        $article['image_alt'] = $article['establishment_name'] . ' health inspection';
        $article['tags'] = $article['tags'] ?: [];
        
        // Add missing geo fields to prevent warnings
        $article['latitude'] = '';
        $article['longitude'] = '';
        
        // Add schema JSON to prevent warnings
        $article['schema_json'] = json_encode([
            '@context' => 'https://schema.org',
            '@type' => 'NewsArticle',
            'headline' => $article['title'],
            'description' => $article['meta_description'],
            'author' => 'CleanKitchens Editorial Team',
            'datePublished' => $article['published_date']
        ]);
        
        return $article;
    }
    
    return null;
}

function getHomepageArticles() {
    $query = '{
        Get {
            Articles(
                limit: 20
                sort: [{path: ["published_date"], order: desc}]
            ) {
                title
                slug
                excerpt
                city
                state
                published_date
                image_url
                establishment_name
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = $response['data']['Get']['Articles'];
        
        // Add formatted time to each article
        foreach ($articles as &$article) {
            $article['formatted_time'] = timeAgo($article['published_date']);
        }
        
        return $articles;
    }
    
    return [];
}

function getFeaturedArticle() {
    $articles = getHomepageArticles();
    return !empty($articles) ? $articles[0] : null;
}

function getRelatedArticles($currentSlug = '', $limit = 3) {
    $query = '{
        Get {
            Articles(
                limit: ' . ($limit + 1) . '
                sort: [{path: ["published_date"], order: desc}]
            ) {
                title
                slug
                city
                state
                published_date
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = $response['data']['Get']['Articles'];
        
        // Remove current article from related
        $related = [];
        foreach ($articles as $article) {
            if ($article['slug'] !== $currentSlug && count($related) < $limit) {
                $related[] = $article;
            }
        }
        
        return $related;
    }
    
    return [];
}

function timeAgo($timestamp) {
    $now = new DateTime();
    $then = new DateTime($timestamp);
    $diff = $now->diff($then);
    
    if ($diff->days == 0) {
        if ($diff->h == 0) {
            return "Just now";
        } elseif ($diff->h == 1) {
            return "1 hour ago";
        } else {
            return $diff->h . " hours ago";
        }
    } elseif ($diff->days == 1) {
        return "Yesterday";
    } elseif ($diff->days < 7) {
        return $diff->days . " days ago";
    } else {
        return $then->format('M j, Y');
    }
}

// Get the inspection image for an article (same for thumbnails and article page)
function getArticleImage($article = null) {
    // Use the single inspection image for all articles
    return '/assets/images/health-inspection-report.jpg';
}

// Simple image optimization function with responsive sizes
function getOptimizedImageUrl($baseUrl, $size, $article = null, $format = 'auto') {
    // Use pre-optimized versions for better performance
    $basePath = '/assets/images/health-inspection-report';
    
    switch ($size) {
        case 'thumb':
            return $basePath . '_thumb.webp'; // 9.7KB WebP for thumbnails
        case 'og':
            return $basePath . '_og.jpg'; // 54KB JPG for social sharing
        case 'page':
            return $basePath . '_page.webp'; // 15KB WebP for feature images
        case 'article':
            return $basePath . '_page.webp'; // 15KB WebP for article images
        default:
            return $basePath . '.jpg'; // Original fallback
    }
}

// Generate WebP image srcset for better performance
function getImageSrcSet($imagePath, $sizes = [400, 800, 1200]) {
    // Use pre-optimized WebP versions for different sizes
    $basePath = '/assets/images/health-inspection-report';
    $srcset = [];
    
    // Map sizes to available optimized versions
    foreach ($sizes as $width) {
        if ($width <= 400) {
            $srcset[] = $basePath . '_thumb.webp 400w'; // 9.7KB
        } elseif ($width <= 800) {
            $srcset[] = $basePath . '_page.webp 800w'; // 15KB
        } else {
            $srcset[] = $basePath . '.webp 1200w'; // 55KB full WebP
        }
    }
    
    return implode(', ', array_unique($srcset));
}
?>