/* CSS Variables for theming */
:root {
    --text-primary: #0a0a0a;
    --text-secondary: #4a4a4a;
    --text-muted: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f0f0f0; /* Darker background for better contrast */
    --bg-card: #ffffff;
    --accent-color: #c41e3a;
    --accent-hover: #a01729;
    --border-color: rgba(0,0,0,0.08);
    --shadow-sm: 0 2px 4px rgba(0,0,0,0.08); /* Slightly stronger shadows */
    --shadow-md: 0 4px 12px rgba(0,0,0,0.1);
    --shadow-lg: 0 8px 24px rgba(0,0,0,0.15);
    --navbar-bg: rgba(255, 255, 255, 0.98);
}

/* Dark mode variables */
@media (prefers-color-scheme: dark) {
    :root:not(.light-mode) {
        --text-primary: #fafafa;
        --text-secondary: #b8b8b8;
        --text-muted: #888888;
        --bg-primary: #1a1a1a;      /* Changed from #0a0a0a - softer dark */
        --bg-secondary: #121212;     /* Changed from #141414 - matches modern dark themes */
        --bg-card: #242424;          /* Changed from #1a1a1a - better card contrast */
        --accent-color: #e74c3c;
        --accent-hover: #c0392b;
        --border-color: rgba(255,255,255,0.08);
        --shadow-sm: 0 2px 4px rgba(0,0,0,0.3);
        --shadow-md: 0 4px 12px rgba(0,0,0,0.4);
        --shadow-lg: 0 8px 24px rgba(0,0,0,0.5);
        --navbar-bg: rgba(26, 26, 26, 0.98);  /* Updated to match new bg */
    }
}

/* Manual dark mode class */
:root.dark-mode {
    --text-primary: #fafafa;
    --text-secondary: #b8b8b8;
    --text-muted: #888888;
    --bg-primary: #1a1a1a;      /* Softer dark */
    --bg-secondary: #121212;     /* Matches modern dark themes */
    --bg-card: #242424;          /* Better card contrast */
    --accent-color: #e74c3c;
    --accent-hover: #c0392b;
    --border-color: rgba(255,255,255,0.08);
    --shadow-sm: 0 2px 4px rgba(0,0,0,0.3);
    --shadow-md: 0 4px 12px rgba(0,0,0,0.4);
    --shadow-lg: 0 8px 24px rgba(0,0,0,0.5);
    --navbar-bg: rgba(26, 26, 26, 0.98);
}

/* Manual light mode class */
:root.light-mode {
    /* Force light theme values */
}

/* Base Reset */
* { 
    margin: 0; 
    padding: 0; 
    box-sizing: border-box; 
}

/* Smooth scrolling */
html {
    scroll-behavior: smooth;
}

/* Typography - System fonts with premium feel */
body { 
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
    line-height: 1.6;
    color: var(--text-primary);
    background: var(--bg-secondary);
    font-size: 16px;
    -webkit-font-smoothing: antialiased;
    -moz-osx-font-smoothing: grayscale;
    transition: background-color 0.3s ease, color 0.3s ease;
}

/* Selection color */
::selection {
    background: var(--accent-color);
    color: white;
}

/* System font stack for display */
:root {
    --font-display: ui-serif, Georgia, 'Times New Roman', serif;
    --font-body: -apple-system, BlinkMacSystemFont, 'Segoe UI', system-ui, sans-serif;
}

/* Mobile Menu Dropdown */
.mobile-nav-menu {
    display: none;
    position: fixed;
    top: 60px;
    left: 0;
    right: 0;
    background: var(--navbar-bg);
    box-shadow: var(--shadow-lg);
    z-index: 99;
}
.mobile-nav-menu.active {
    display: block;
}
.mobile-nav-menu a {
    display: block;
    padding: 16px 16px;
    color: var(--text-secondary);
    text-decoration: none;
    font-weight: 500;
    border-bottom: 1px solid var(--border-color);
    transition: color 0.2s ease, padding-left 0.2s ease;
}
.mobile-nav-menu a:hover {
    background: var(--bg-secondary);
    color: var(--accent-color);
    padding-left: 24px;
}
.mobile-nav-menu a:last-child {
    border-bottom: none;
}

/* Container */
.container { 
    max-width: 1140px; 
    margin: 0 auto; 
    padding: 0 16px; 
}

/* Navigation with optimized performance */
.navbar {
    background: var(--navbar-bg);
    box-shadow: 0 1px 0 var(--border-color);
    padding: 12px 0;
    position: sticky;
    top: 0;
    z-index: 100;
    transition: background-color 0.3s ease, box-shadow 0.3s ease;
}
.nav-wrapper {
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.navbar-brand {
    font-size: 24px;
    font-weight: 700;
    color: var(--accent-color);
    text-decoration: none;
    transition: opacity 0.2s ease;
    display: inline-block;
    padding: 8px;
    min-height: 44px;
}
.navbar-brand:hover {
    opacity: 0.8;
}
.nav-links {
    display: none;
}
.nav-links a {
    position: relative;
    color: var(--text-secondary);
    text-decoration: none;
    font-weight: 500;
    font-size: 15px;
    transition: color 0.2s ease;
}
.nav-links a:hover {
    color: var(--accent-color);
}
.nav-links a::after {
    content: '';
    position: absolute;
    bottom: -4px;
    left: 0;
    right: 0;
    height: 2px;
    background: var(--accent-color);
    transform: scaleX(0);
    transition: transform 0.3s ease;
}
.nav-links a:hover::after {
    transform: scaleX(1);
}

.mobile-menu {
    width: 28px;
    height: 20px;
    position: relative;
    cursor: pointer;
    -webkit-tap-highlight-color: transparent;
}
.mobile-menu span {
    display: block;
    height: 3px;
    background: var(--text-primary);
    margin-bottom: 5px;
    border-radius: 2px;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1), opacity 0.3s ease;
}
.mobile-menu span:last-child {
    margin-bottom: 0;
}
.mobile-menu.active span:nth-child(1) {
    transform: rotate(45deg) translate(5px, 5px);
}
.mobile-menu.active span:nth-child(2) {
    opacity: 0;
}
.mobile-menu.active span:nth-child(3) {
    transform: rotate(-45deg) translate(7px, -6px);
}

/* Theme toggle button */
.theme-toggle {
    background: transparent;
    border: 2px solid var(--border-color);
    border-radius: 20px;
    padding: 6px 12px;
    cursor: pointer;
    font-size: 16px;
    transition: border-color 0.3s ease, color 0.3s ease, transform 0.3s ease;
    color: var(--text-primary);
    margin-left: 16px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 44px;
    min-height: 44px;
}
.theme-toggle:hover {
    border-color: var(--accent-color);
    color: var(--accent-color);
    transform: translateY(-2px);
}

/* Mobile theme toggle in menu */
@media (max-width: 767px) {
    .theme-toggle {
        display: none; /* Hide desktop toggle on mobile */
    }
}

/* Hero - Minimal news-style header */
.hero {
    background: transparent;
    color: var(--text-primary);
    padding: 24px 0;
    margin-bottom: 24px;
    border-bottom: 2px solid var(--border-color);
}
.hero h1 {
    font-family: var(--font-body);
    font-size: 24px;
    line-height: 1.3;
    margin-bottom: 8px;
    font-weight: 600;
    letter-spacing: -0.02em;
}
.hero p {
    font-size: 16px;
    line-height: 1.5;
    color: var(--text-secondary); /* Better contrast for accessibility */
}

/* Section Headers */
.section {
    margin-bottom: 48px;
}
.section-header {
    margin-bottom: 24px;
    padding-bottom: 16px;
    border-bottom: 2px solid var(--border-color);
}
.section-title {
    font-family: var(--font-display);
    font-size: 28px;
    font-weight: 700;
    color: var(--text-primary);
    letter-spacing: -0.02em;
}

/* News Section and Grid - CLS Prevention */
.news-section {
    margin-bottom: 3rem;
    /* Prevent layout shifts with stable container */
    min-height: 800px;
    contain: layout;
}

.articles-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1.5rem;
    /* Prevent layout shifts during load */
    min-height: 700px;
    contain: layout;
}

/* Cards - Optimized animations */
.article-card {
    background: var(--bg-card);
    border-radius: 8px;
    overflow: hidden;
    box-shadow: var(--shadow-sm);
    margin-bottom: 20px;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1), box-shadow 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid var(--border-color);
    /* Prevent layout shifts */
    min-height: 380px;
    display: flex;
    flex-direction: column;
    contain: layout style;
}
.article-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
    border-color: transparent;
}
.article-image {
    width: 100%;
    height: 180px;
    background: var(--bg-secondary);
    position: relative;
    overflow: hidden;
    /* Prevent layout shifts */
    min-height: 180px;
    flex-shrink: 0;
    aspect-ratio: 4/3;
    contain: layout;
}
.article-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    display: block;
    /* Remove hover animation that was causing movement */
}
.article-content {
    padding: 20px;
    /* Fill remaining space in flexbox to prevent layout shifts */
    flex: 1;
    display: flex;
    flex-direction: column;
    justify-content: space-between;
}
.article-meta {
    font-size: 14px;
    color: var(--text-muted);
    margin-bottom: 8px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-weight: 500;
}
.article-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
    line-height: 1.4;
    transition: color 0.2s ease;
}
.article-card:hover .article-title {
    color: var(--accent-color);
}
.article-excerpt {
    font-size: 15px;
    color: var(--text-secondary);
    margin-bottom: 16px;
    line-height: 1.5;
}
.article-link {
    color: var(--accent-color);
    font-weight: 500;
    text-decoration: none;
    font-size: 15px;
    background-image: linear-gradient(var(--accent-color), var(--accent-color));
    background-size: 0% 2px;
    background-position: 0 100%;
    background-repeat: no-repeat;
    transition: background-size 0.3s ease;
}
.article-link:hover {
    background-size: 100% 2px;
}

/* Featured Article - Premium */
.featured-article {
    background: var(--bg-card);
    border-radius: 12px;
    overflow: hidden;
    box-shadow: var(--shadow-md);
    margin-bottom: 48px;
    transition: box-shadow 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid var(--border-color);
}
.featured-article:hover {
    box-shadow: var(--shadow-lg);
}
.featured-image {
    width: 100%;
    height: 220px;
    background: var(--bg-secondary);
    position: relative;
    overflow: hidden;
}
.featured-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    display: block;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}
.featured-article:hover .featured-image img {
    transform: scale(1.05);
}
.featured-content {
    padding: 24px;
}

/* Fix mobile featured article spacing */
@media (max-width: 767px) {
    .featured-article {
        margin-bottom: 24px !important;
        display: block !important;
    }
    
    .featured-image {
        height: 200px !important;
        width: 100% !important;
        margin-bottom: 0 !important;
    }
    
    .featured-content {
        padding: 12px !important;
        margin-top: 0 !important;
    }
    
    .featured-badge {
        margin-bottom: 4px !important;
    }
    
    .featured-content .article-title {
        margin: 4px 0 !important;
        font-size: 1.25rem !important;
        line-height: 1.3 !important;
    }
    
    .featured-content .article-meta {
        margin: 4px 0 !important;
        font-size: 0.875rem !important;
    }
    
    .featured-content .article-excerpt {
        margin: 8px 0 !important;
        line-height: 1.4 !important;
    }
    
    .featured-content .article-link {
        margin-top: 8px !important;
        display: inline-block;
    }
}
.featured-badge {
    display: inline-block;
    background: var(--accent-color);
    color: white;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    margin-bottom: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}
.featured-content .article-title {
    font-family: var(--font-display);
    font-size: 24px;
    letter-spacing: -0.02em;
    margin-bottom: 12px;
}

/* Grid System - Mobile First */
.row {
    display: flex;
    flex-wrap: wrap;
    margin: 0 -8px;
}
.col {
    width: 100%;
    padding: 0 8px;
}

/* Footer - Dark theme */
footer {
    background: #1a1a1a;
    color: #cccccc;
    padding: 48px 0 32px;
    margin-top: 64px;
    font-size: 14px;
    transition: background-color 0.3s ease;
    text-align: center;
}

/* Dark mode footer */
:root.dark-mode footer {
    background: #0f0f0f;    /* Slightly darker than body for depth */
    border-top: 1px solid rgba(255,255,255,0.1);
}

@media (prefers-color-scheme: dark) {
    :root:not(.light-mode) footer {
        background: #0f0f0f;    /* Slightly darker than body for depth */
        border-top: 1px solid rgba(255,255,255,0.1);
    }
}

.footer-links {
    display: flex;
    flex-wrap: wrap;
    gap: 24px;
    margin-bottom: 24px;
    justify-content: center;
}
.footer-links a {
    color: #cccccc;
    text-decoration: none;
    transition: color 0.2s ease;
}
.footer-links a:hover {
    color: white;
}
.footer-copyright {
    color: #999999;
    font-size: 13px;
}
.footer-copyright a {
    color: #999999;
    text-decoration: underline;
}
.footer-copyright a:hover {
    color: #cccccc;
}

/* Header social icons */
.social-links {
    display: none;
    align-items: center;
    gap: 12px;
    margin-left: 16px;
    padding-left: 16px;
    border-left: 1px solid var(--border-color);
}
.social-links a {
    color: var(--text-secondary);
    font-size: 18px;
    text-decoration: none;
    transition: color 0.2s ease, transform 0.2s ease, background-color 0.2s ease;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    font-weight: bold;
}
.social-links a:hover {
    color: var(--accent-color);
    background: var(--bg-secondary);
    transform: translateY(-2px);
}

/* Footer social section */
.footer-social {
    margin-bottom: 24px;
    padding-bottom: 24px;
    border-bottom: 1px solid rgba(255,255,255,0.1);
}
.footer-social-title {
    font-size: 14px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: #999999;
    margin-bottom: 16px;
}
.footer-social-links {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}
.footer-social-links a {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 8px 16px;
    background: rgba(255, 255, 255, 0.1);
    color: #cccccc;
    text-decoration: none;
    border-radius: 24px;
    font-size: 14px;
    transition: background-color 0.2s ease, color 0.2s ease, transform 0.2s ease, border-color 0.2s ease;
    border: 1px solid rgba(255, 255, 255, 0.2);
}
.footer-social-links a:hover {
    background: var(--accent-color);
    color: white;
    border-color: var(--accent-color);
    transform: translateY(-2px);
}
.footer-social-links .social-icon {
    font-size: 16px;
    font-weight: bold;
}

/* Article Page Styles */
.article-wrapper {
    background: var(--bg-primary);
    margin-top: 24px;
    box-shadow: var(--shadow-sm);
    border-radius: 8px;
    overflow: hidden;
    transition: background-color 0.3s ease, box-shadow 0.3s ease;
}
.article-header {
    padding: 32px 24px 24px;
}
.article-meta {
    font-size: 14px;
    color: var(--text-muted);
    margin-bottom: 16px;
    letter-spacing: 0.5px;
    text-transform: uppercase;
    font-weight: 500;
}
.article-meta span {
    margin-right: 16px;
}
.article-title {
    font-family: var(--font-display);
    font-size: 32px;
    font-weight: 700;
    line-height: 1.2;
    color: var(--text-primary);
    margin-bottom: 16px;
    letter-spacing: -0.02em;
}
.article-image {
    width: 100%;
    margin-bottom: 24px;
    overflow: hidden;
    position: relative;
    background: var(--bg-secondary);
    min-height: 200px;
}
.article-image img {
    width: 100%;
    height: auto;
    display: block;
}

/* Fix mobile white space */
@media (max-width: 767px) {
    .article-image {
        margin-bottom: 16px;
        min-height: 180px;
    }
}

/* Article content with optimal reading experience */
.article-content {
    max-width: 680px;
    margin: 0 auto;
    padding: 0 24px 48px;
    font-size: 18px;
    line-height: 1.8;
    color: var(--text-secondary);
}
.article-content p {
    margin-bottom: 24px;
}

/* Drop cap removed for cleaner, more modern look */

/* Premium link styling */
.article-content a {
    color: var(--accent-color);
    text-decoration: none;
    font-weight: 500;
    background-image: linear-gradient(var(--accent-color), var(--accent-color));
    background-size: 0% 2px;
    background-position: 0 100%;
    background-repeat: no-repeat;
    transition: background-size 0.3s ease;
}
.article-content a:hover {
    background-size: 100% 2px;
}

/* Elegant section divider */
.article-content hr {
    border: none;
    text-align: center;
    margin: 48px 0;
}
.article-content hr::before {
    content: "• • •";
    color: var(--text-muted);
    letter-spacing: 8px;
    font-size: 18px;
}

/* Tags with premium styling */
.article-tags {
    margin: 0 24px 32px;
    padding-top: 24px;
    border-top: 1px solid var(--border-color);
    max-width: 680px;
    margin-left: auto;
    margin-right: auto;
}
.tag {
    display: inline-block;
    background: var(--bg-secondary);
    color: var(--text-secondary);
    padding: 12px 16px;
    border-radius: 20px;
    min-height: 44px;
    font-size: 14px;
    margin-right: 8px;
    margin-bottom: 8px;
    text-decoration: none;
    transition: border-color 0.2s ease, color 0.2s ease, transform 0.2s ease;
    border: 1px solid transparent;
}
.tag:hover {
    border-color: var(--accent-color);
    color: var(--accent-color);
    transform: translateY(-2px);
}

/* Related Articles with card animations */
.related-section {
    margin-top: 48px;
    padding-top: 32px;
}
.section-title {
    font-family: var(--font-display);
    font-size: 28px;
    font-weight: 700;
    margin-bottom: 24px;
    letter-spacing: -0.02em;
}
.related-card {
    background: var(--bg-card);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 16px;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1), box-shadow 0.3s cubic-bezier(0.4, 0, 0.2, 1), background-color 0.3s ease;
}
.related-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
    border-color: transparent;
}
.related-card h3 {
    font-size: 18px;
    margin-bottom: 8px;
    line-height: 1.4;
    transition: color 0.2s ease;
}
.related-card a {
    color: var(--text-primary);
    text-decoration: none;
}
.related-card:hover h3 {
    color: var(--accent-color);
}
.related-meta {
    font-size: 14px;
    color: var(--text-muted);
}

/* Share Buttons - Floating Vertical on Desktop */
.share-section {
    position: fixed;
    left: 20px;
    top: 50%;
    transform: translateY(-50%);
    background: var(--bg-card);
    border-radius: 12px;
    padding: 16px 12px;
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    transition: opacity 0.3s ease;
    z-index: 50;
}

.share-title {
    display: none; /* Hide title in floating version */
}

.share-buttons {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

/* Share button base styles */
.share-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 44px;
    height: 44px;
    padding: 0;
    border-radius: 50%;
    text-decoration: none !important;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    border: none;
    position: relative;
    overflow: hidden;
}

.share-btn:hover {
    transform: translateY(-2px) scale(1.1);
    box-shadow: var(--shadow-lg);
}

/* Force white color on all text elements */
.share-btn,
.share-btn *,
.share-btn span {
    color: #FFFFFF !important;
}

/* Icon styling */
.share-icon {
    font-size: 20px;
    font-weight: bold;
    color: #FFFFFF !important;
    display: block;
    line-height: 1;
}

.share-facebook .share-icon {
    font-family: -apple-system, BlinkMacSystemFont, sans-serif;
}

.share-text {
    display: none; /* Hidden on desktop */
    color: #FFFFFF !important;
    font-weight: 700;
}

/* Solid backgrounds with darker colors for better contrast */
.share-facebook {
    background-color: #0A4C8F !important; /* Much darker blue */
    color: #FFFFFF !important;
}

.share-facebook:hover {
    background-color: #073A6C !important;
}

.share-twitter {
    background-color: #000000 !important;
    color: #FFFFFF !important;
}

.share-twitter:hover {
    background-color: #1A1A1A !important;
}

.share-email {
    background-color: #333333 !important; /* Darker gray */
    color: #FFFFFF !important;
}

.share-email:hover {
    background-color: #1A1A1A !important;
}

/* Hide floating buttons until article starts */
@media (min-width: 1200px) {
    .share-section {
        display: block;
        opacity: 1; /* Always visible for now */
        pointer-events: all;
    }
}

/* Mobile - Horizontal at bottom of article */
@media (max-width: 1199px) {
    .share-section {
        position: static !important; /* Override fixed positioning */
        left: auto;
        top: auto;
        transform: none;
        margin: 48px auto;
        padding: 24px;
        max-width: 680px;
        text-align: center;
        background: var(--bg-secondary);
        box-shadow: none;
        opacity: 1 !important; /* Always visible on mobile */
        pointer-events: all !important;
    }
    
    .share-title {
        display: block;
        font-size: 16px;
        font-weight: 600;
        margin-bottom: 16px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        color: var(--text-secondary); /* Better contrast */
    }
    
    .share-buttons {
        flex-direction: row;
        justify-content: center;
        gap: 12px;
    }
    
    .share-btn {
        width: auto;
        height: auto;
        padding: 10px 20px;
        border-radius: 24px;
        font-size: 14px;
        gap: 8px;
        font-weight: 500; /* Add weight for better contrast */
    }
    
    /* Hide icon and show text on mobile */
    .share-icon {
        display: none;
    }
    
    .share-text {
        display: inline;
    }
}

/* Mobile placement at end of article */
@media (max-width: 767px) {
    .share-section {
        margin: 0 -24px; /* Full width on mobile */
        border-radius: 0;
        border-left: none;
        border-right: none;
    }
}

/* Full width on small mobile */
@media (max-width: 767px) {
    .share-section {
        margin-left: -16px;
        margin-right: -16px;
        border-radius: 0;
        border-left: none;
        border-right: none;
    }
    
    .article-content {
        padding: 0 16px 32px;
    }
    
    .article-header {
        padding: 24px 16px;
    }
    
    .article-tags {
        margin: 0 16px 24px;
    }
}

/* Static Page */
.static-content {
    background: var(--bg-primary);
    padding: 48px;
    margin: 32px 0;
    border-radius: 8px;
    max-width: 800px;
    margin-left: auto;
    margin-right: auto;
}

.static-content h1 {
    font-family: var(--font-display);
    font-size: 36px;
    margin-bottom: 32px;
    letter-spacing: -0.02em;
    color: var(--text-primary);
}

.static-content h2 {
    font-size: 24px;
    margin-top: 32px;
    margin-bottom: 16px;
    color: var(--text-primary);
}

.static-content p {
    font-size: 18px;
    line-height: 1.7;
    margin-bottom: 20px;
    color: var(--text-secondary);
}

.static-content ul {
    font-size: 18px;
    line-height: 1.7;
    color: var(--text-secondary);
}

.static-content a {
    color: var(--accent-color);
    text-decoration: none;
}

.static-content a:hover {
    text-decoration: underline;
}

/* Mobile adjustments for footer social */
@media (max-width: 767px) {
    .footer-social-links {
        justify-content: center;
    }
    .footer-social-title {
        text-align: center;
    }
    .static-content {
        padding: 32px 24px;
    }
}

/* Tablet and up */
@media (min-width: 768px) {
    .container { padding: 0 24px; }
    .navbar { padding: 16px 0; }
    .nav-links {
        display: flex;
        gap: 32px;
        align-items: center;
    }
    .social-links {
        display: flex;
    }
    .mobile-menu { display: none; }
    .mobile-nav-menu { display: none !important; }
    
    /* Minimal hero on larger screens */
    .hero { 
        padding: 32px 0; 
        margin-bottom: 40px; 
    }
    .hero h1 { 
        font-size: 28px; 
        margin-bottom: 12px;
    }
    .hero p { 
        font-size: 18px; 
    }
    
    .section-title { font-size: 32px; }
    
    .col-md-4 { width: 33.333%; }
    .col-md-6 { width: 50%; }
    .col-md-8 { width: 66.666%; }
    
    .article-card { margin-bottom: 0; }
    .article-card .article-image { height: 240px; } /* Increase height for better desktop display */
    .article-wrapper .article-image { height: auto; max-height: none; } /* Full size for article pages */
    .article-title { font-size: 20px; }
    .featured-content .article-title { font-size: 32px; }
    
    .featured-article { display: flex; }
    .featured-image { width: 66.666%; height: 400px; }
    .featured-content { width: 33.333%; padding: 32px; display: flex; flex-direction: column; justify-content: center; }
    
    .article-header { padding: 48px 48px 32px; }
    .article-title { font-size: 42px; }
    .article-content { 
        font-size: 19px; 
        padding: 0 48px 64px;
    }
}

/* Desktop */
@media (min-width: 1024px) {
    .hero h1 { font-size: 32px; }
    .hero p { font-size: 18px; }
    .section-title { font-size: 36px; }
    .featured-content .article-title { font-size: 36px; }
    .article-title { font-size: 48px; line-height: 1.1; }
    .article-content { font-size: 20px; line-height: 1.8; }
    .static-content h1 { font-size: 48px; }
}

/* Simple loading placeholder - no animation */
.image-loading {
    background: var(--bg-secondary);
}

/* Skip link for accessibility */
.skip-link {
    position: absolute;
    top: -40px;
    left: 0;
    background: var(--accent-color);
    color: white;
    padding: 8px;
    text-decoration: none;
    border-radius: 0 0 4px 0;
}
.skip-link:focus {
    top: 0;
}