#!/usr/bin/env python3
"""
Download better, professional stock images from Unsplash for CleanKitchens
"""

import os
import requests
import time
from pathlib import Path

# Unsplash API (no key needed for basic usage with their official URLs)
BASE_URL = "https://source.unsplash.com"

# Image categories and search terms for more professional, neutral images
IMAGE_CATEGORIES = {
    "restaurants": [
        "restaurant-interior",
        "clean-restaurant-kitchen",
        "restaurant-dining-room",
        "restaurant-exterior",
        "restaurant-table-setting",
        "professional-kitchen",
        "chef-cooking",
        "restaurant-service"
    ],
    "food_safety": [
        "food-preparation",
        "kitchen-hygiene",
        "chef-washing-hands",
        "clean-kitchen",
        "food-storage",
        "restaurant-refrigerator",
        "food-thermometer",
        "kitchen-cleaning"
    ],
    "inspections": [
        "clipboard-checklist",
        "health-inspector",
        "quality-control",
        "food-inspection",
        "restaurant-review",
        "safety-checklist",
        "professional-assessment",
        "workplace-inspection"
    ],
    "general": [
        "restaurant-business",
        "food-service",
        "hospitality-industry",
        "commercial-kitchen",
        "restaurant-staff",
        "food-industry",
        "restaurant-management",
        "dining-experience"
    ],
    "chains": [
        "fast-food-restaurant",
        "coffee-shop",
        "pizza-restaurant",
        "burger-restaurant",
        "sandwich-shop",
        "bakery-cafe",
        "casual-dining",
        "restaurant-franchise"
    ]
}

def download_image(search_term, filename, size="1920x1080"):
    """Download a single image from Unsplash"""
    # Using Unsplash Source API for direct image URLs
    url = f"{BASE_URL}/1920x1080/?{search_term}"
    
    try:
        response = requests.get(url, allow_redirects=True, timeout=30)
        if response.status_code == 200:
            with open(filename, 'wb') as f:
                f.write(response.content)
            print(f"✓ Downloaded: {filename}")
            return True
        else:
            print(f"✗ Failed to download {search_term}: {response.status_code}")
            return False
    except Exception as e:
        print(f"✗ Error downloading {search_term}: {e}")
        return False

def main():
    # Create base directory structure
    base_dir = Path("/var/www/twin-digital-media/public_html/_sites/cleankitchens/assets/images/stock")
    base_dir.mkdir(exist_ok=True)
    
    print("Downloading professional stock images from Unsplash...")
    print("=" * 60)
    
    for category, terms in IMAGE_CATEGORIES.items():
        # Create category directory
        category_dir = base_dir / category
        category_dir.mkdir(exist_ok=True)
        
        print(f"\nCategory: {category}")
        print("-" * 40)
        
        for i, term in enumerate(terms, 1):
            filename = category_dir / f"{category}_{i}.jpg"
            download_image(term, filename)
            # Be respectful to Unsplash's servers
            time.sleep(2)
    
    print("\n" + "=" * 60)
    print("Download complete!")
    print(f"Images saved to: {base_dir}")
    
    # Create a metadata file
    metadata_file = base_dir / "image_metadata.json"
    import json
    metadata = {
        "source": "Unsplash",
        "license": "Free to use under Unsplash License",
        "categories": IMAGE_CATEGORIES,
        "download_date": time.strftime("%Y-%m-%d"),
        "notes": "Professional, neutral stock photos for news reporting"
    }
    
    with open(metadata_file, 'w') as f:
        json.dump(metadata, f, indent=2)
    
    print(f"Metadata saved to: {metadata_file}")

if __name__ == "__main__":
    main()