#!/usr/bin/env python3
"""
Generate a professional restaurant inspection image using DALL-E 3
"""

import os
from openai import OpenAI
import requests
from PIL import Image
from io import BytesIO

# Initialize OpenAI client
client = OpenAI(api_key=os.getenv('OPENAI_API_KEY'))

def generate_inspection_image():
    """Generate a professional restaurant inspection themed image"""
    
    prompts = [
        # Option 1: Clipboard
        "A professional clipboard with a restaurant health inspection checklist, clean modern design, photorealistic, white background, official looking document with checkboxes and 'HEALTH INSPECTION' header in bold letters, pen attached to clipboard, high quality stock photo style",
        
        # Option 2: Yellow warning sign
        "A yellow caution-style standing sign that says 'HEALTH INSPECTION IN PROGRESS' in bold black letters, similar to wet floor signs, professional photography, clean white background, industrial safety sign design",
        
        # Option 3: Kitchen door sign
        "A professional sign hanging on a stainless steel commercial kitchen door that reads 'HEALTH INSPECTION' in bold letters, clean modern design, photorealistic, restaurant kitchen setting, official looking placard"
    ]
    
    # Let's generate all three and you can pick the best one
    for i, prompt in enumerate(prompts, 1):
        print(f"Generating image option {i}...")
        
        try:
            response = client.images.generate(
                model="dall-e-3",
                prompt=prompt,
                size="1792x1024",  # Wide format good for web
                quality="hd",
                n=1
            )
            
            image_url = response.data[0].url
            
            # Download the image
            img_response = requests.get(image_url)
            img = Image.open(BytesIO(img_response.content))
            
            # Save in multiple formats
            base_path = f'/var/www/twin-digital-media/public_html/_sites/cleankitchens/assets/images/inspection_option_{i}'
            
            # Save as JPG for social sharing
            img.save(f'{base_path}.jpg', 'JPEG', quality=90)
            print(f"✅ Saved: {base_path}.jpg")
            
            # Save as WebP for web display
            img.save(f'{base_path}.webp', 'WEBP', quality=85)
            print(f"✅ Saved: {base_path}.webp")
            
            # Create thumbnail version (600x400)
            thumb = img.copy()
            thumb.thumbnail((600, 400), Image.Resampling.LANCZOS)
            thumb.save(f'{base_path}_thumb.jpg', 'JPEG', quality=85)
            thumb.save(f'{base_path}_thumb.webp', 'WEBP', quality=80)
            print(f"✅ Saved thumbnail versions")
            
        except Exception as e:
            print(f"❌ Error generating image {i}: {e}")
    
    print("\n✨ Complete! Three inspection image options generated.")
    print("View them at:")
    print("  /assets/images/inspection_option_1.jpg (Clipboard)")
    print("  /assets/images/inspection_option_2.jpg (Yellow Sign)")
    print("  /assets/images/inspection_option_3.jpg (Kitchen Door)")

if __name__ == "__main__":
    # Check for API key
    if not os.getenv('OPENAI_API_KEY'):
        print("⚠️  Please set OPENAI_API_KEY environment variable")
        print("   export OPENAI_API_KEY='your-key-here'")
        exit(1)
    
    generate_inspection_image()